package tech.ostack.kform.schemas

import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlin.reflect.KType
import kotlinx.coroutines.flow.Flow
import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.AbsolutePathFragment
import tech.ostack.kform.CollectionSchema
import tech.ostack.kform.CollectionState
import tech.ostack.kform.ComputedValue
import tech.ostack.kform.ParentSchema
import tech.ostack.kform.ParentState
import tech.ostack.kform.Schema
import tech.ostack.kform.SchemaEventsBus
import tech.ostack.kform.SchemaInfo
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.ValueEvent
import tech.ostack.kform.ValueInfo
import tech.ostack.kform.schemaInfo
import tech.ostack.kform.validations.MatchesComputedValue

/**
 * A schema representing a [computed value][computedValue] of type [T]: a form value that should
 * equal the result of evaluating a given computation over other values of the form. A common
 * example of a computed value would be a field whose value should equal the sum of other form
 * values.
 *
 * Example usage:
 * ```kotlin
 * data class ItemPurchase(
 *     val price: Double?,
 *     val amount: Int?,
 *     val tax: Float?,
 *     val total: Double
 * )
 *
 * object Total : ComputedValue<Double> {
 *     private val ComputedValueContext.price: Double? by dependency("../price")
 *     private val ComputedValueContext.amount: Int? by dependency("../amount")
 *     private val ComputedValueContext.tax: Float? by dependency("../tax")
 *
 *     override suspend fun ComputedValueContext.compute(): Double {
 *         val totalPrice = (price ?: 0.0) * (amount ?: 0)
 *         return totalPrice + totalPrice * (tax ?: 0f)
 *     }
 * }
 *
 * val itemPurchaseSchema = ClassSchema {
 *     ItemPurchase::price { NullableSchema { DoubleSchema() } }
 *     ItemPurchase::amount { NullableSchema { IntSchema() } }
 *     ItemPurchase::tax { NullableSchema { FloatSchema() } }
 *     ItemPurchase::total { ComputedSchema(Total) { DoubleSchema() } }
 * }
 * ```
 *
 * This schema will defer its implementation to the provided inner [innerSchema].
 *
 * This schema automatically includes a validation which checks that the form value equals the
 * result of evaluating the provided [computed value][computedValue]. A [valueMismatchCode] and
 * [valueMismatchSeverity] may be provided to specify the code and severity of the issue emitted
 * when the form value doesn't match the result of the computation. The code defaults to
 * [MatchesComputedValue.DEFAULT_CODE] and the severity to [ValidationIssueSeverity.Error].
 */
public interface ComputedSchema<T> : Schema<T> {
    /** Computed value represented by this schema. */
    public val computedValue: ComputedValue<T>

    /** Inner schema of the value being computed. */
    public val innerSchema: Schema<T>

    /** Validation issue code to emit when the value doesn't match the result of the computation. */
    public val valueMismatchCode: String

    /**
     * Severity of the validation issue to emit when the value doesn't match the result of the
     * computation.
     */
    public val valueMismatchSeverity: ValidationIssueSeverity

    public companion object {
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun <T> invoke(
            computedValue: ComputedValue<T>,
            valueMismatchCode: String = MatchesComputedValue.DEFAULT_CODE,
            valueMismatchSeverity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
        ): ComputedSchema<T?> =
            ComputedSimpleSchema(
                computedValue,
                AnySchema<T>(),
                valueMismatchCode,
                valueMismatchSeverity,
            )

        /** Function that builds a schema representing computed values of type [T]. */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        @Suppress("UNCHECKED_CAST")
        public operator fun <T> invoke(
            computedValue: ComputedValue<T>,
            valueMismatchCode: String = MatchesComputedValue.DEFAULT_CODE,
            valueMismatchSeverity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
            @BuilderInference builder: () -> Schema<T>,
        ): ComputedSchema<T> =
            when (val schema = builder()) {
                is CollectionSchema<*, *> -> {
                    schema as CollectionSchema<T, Any?>
                    ComputedCollectionSchema(
                        computedValue,
                        schema,
                        valueMismatchCode,
                        valueMismatchSeverity,
                    )
                }
                is ParentSchema<*> -> {
                    schema as ParentSchema<T>
                    ComputedParentSchema<T, Any?, ParentSchema<T>>(
                        computedValue,
                        schema,
                        valueMismatchCode,
                        valueMismatchSeverity,
                    )
                }
                else ->
                    ComputedSimpleSchema(
                        computedValue,
                        schema,
                        valueMismatchCode,
                        valueMismatchSeverity,
                    )
            }
    }
}

/**
 * Schema representing computed simple values of type [T] represented by schemas of type [TSchema].
 */
internal open class ComputedSimpleSchema<T, TSchema : Schema<T>>(
    override val computedValue: ComputedValue<T>,
    override val innerSchema: TSchema,
    override val valueMismatchCode: String,
    override val valueMismatchSeverity: ValidationIssueSeverity,
) : Schema<T>, ComputedSchema<T> {
    init {
        require(schemaInfo(innerSchema).all { it.schema !is ComputedSchema }) {
            "Cannot create nested computed schemas."
        }
    }

    override val typeInfo: TypeInfo = innerSchema.typeInfo

    override val validations: List<Validation<T>> = buildList {
        add(MatchesComputedValue(computedValue, valueMismatchCode, valueMismatchSeverity))
        addAll(innerSchema.validations)
    }

    override val initialValue: T = innerSchema.initialValue

    override suspend fun clone(value: T): T = innerSchema.clone(value)

    override fun assignableTo(type: KType): Boolean = innerSchema.assignableTo(type)

    /**
     * Changes [event] when its path is [path] (the path of this schema) so that it references this
     * schema instead of [innerSchema]. Returns [event] when nothing needs to be changed or the new
     * (replaced) event otherwise.
     */
    protected open fun replaceEvent(path: AbsolutePath, event: ValueEvent<*>): ValueEvent<*> {
        if (path != event.path) {
            return event
        }
        @Suppress("UNCHECKED_CAST")
        return when (event) {
            is ValueEvent.Init<*> -> ValueEvent.Init(event.value as T, path, this)
            is ValueEvent.Change<*> ->
                ValueEvent.Change(event.oldValue as T, event.value as T, path, this)
            is ValueEvent.Destroy<*> -> ValueEvent.Destroy(event.oldValue as T, path, this)
            else -> event
        }
    }

    /**
     * Runs [fn] with a custom events bus where all events emitted to [eventsBus] are mapped through
     * [replaceEvent]: this makes sure that no schema event whose path is [path] (the path of this
     * schema) references a schema that isn't this one.
     */
    protected suspend inline fun withReplacedEvents(
        path: AbsolutePath,
        eventsBus: SchemaEventsBus,
        crossinline fn: suspend (eventsBus: SchemaEventsBus) -> Unit,
    ) =
        fn(
            object : SchemaEventsBus {
                override suspend fun emit(event: ValueEvent<*>) {
                    eventsBus.emit(replaceEvent(path, event))
                }
            }
        )

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (T) -> Unit,
    ) =
        withReplacedEvents(path, eventsBus) { replacedEventsBus ->
            innerSchema.init(path, fromValue, replacedEventsBus, setValue)
        }

    override suspend fun change(
        path: AbsolutePath,
        value: T,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (T) -> Unit,
    ) =
        withReplacedEvents(path, eventsBus) { replacedEventsBus ->
            innerSchema.change(path, value, intoValue, replacedEventsBus, setValue)
        }

    override suspend fun destroy(
        path: AbsolutePath,
        value: T,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (T) -> Unit,
    ) =
        withReplacedEvents(path, eventsBus) { replacedEventsBus ->
            innerSchema.destroy(path, value, replacedEventsBus, removeValue)
        }
}

/**
 * Schema representing computed parent values of type [T] represented by schemas of type [TSchema].
 */
internal open class ComputedParentSchema<T, TChildren, TSchema : ParentSchema<T>>(
    computedValue: ComputedValue<T>,
    innerSchema: TSchema,
    valueMismatchCode: String,
    valueMismatchSeverity: ValidationIssueSeverity,
) :
    ComputedSimpleSchema<T, TSchema>(
        computedValue,
        innerSchema,
        valueMismatchCode,
        valueMismatchSeverity,
    ),
    ParentSchema<T> {
    override val supportsConcurrentSets: Boolean = innerSchema.supportsConcurrentSets

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        innerSchema.isValidChildSchemaFragment(fragment)

    @Suppress("UNCHECKED_CAST")
    override fun childrenSchemas(
        path: AbsolutePath,
        queriedPath: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<SchemaInfo<TChildren>> =
        innerSchema.childrenSchemas(path, queriedPath, fragment) as Sequence<SchemaInfo<TChildren>>

    override suspend fun isValidChildFragment(value: T, fragment: AbsolutePathFragment): Boolean =
        innerSchema.isValidChildFragment(value, fragment)

    @Suppress("UNCHECKED_CAST")
    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: T,
        fragment: AbsolutePathFragment,
    ): Flow<ValueInfo<TChildren>> =
        innerSchema.children(path, schemaPath, value, fragment) as Flow<ValueInfo<TChildren>>

    override suspend fun isValidSetFragment(value: T, fragment: AbsolutePathFragment): Boolean =
        innerSchema.isValidSetFragment(value, fragment)

    override suspend fun set(
        path: AbsolutePath,
        value: T,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus,
    ) =
        withReplacedEvents(path, eventsBus) { replacedEventsBus ->
            innerSchema.set(path, value, fragment, childValue, replacedEventsBus)
        }

    override fun childrenStatesContainer(): ParentState = innerSchema.childrenStatesContainer()
}

/**
 * Schema representing computed collections of type [T] represented by schemas of type [TSchema]
 * with children of type [TChildren].
 */
internal open class ComputedCollectionSchema<
    T,
    TChildren,
    TSchema : CollectionSchema<T, TChildren>,
>(
    computedValue: ComputedValue<T>,
    innerSchema: TSchema,
    valueMismatchCode: String,
    valueMismatchSeverity: ValidationIssueSeverity,
) :
    ComputedParentSchema<T, TChildren, TSchema>(
        computedValue,
        innerSchema,
        valueMismatchCode,
        valueMismatchSeverity,
    ),
    CollectionSchema<T, TChildren> {
    override val supportsConcurrentRemoves: Boolean = innerSchema.supportsConcurrentRemoves

    // Replace additional actions (`Add` and `Remove`)
    override fun replaceEvent(path: AbsolutePath, event: ValueEvent<*>): ValueEvent<*> {
        if (path != event.path) {
            return event
        }
        @Suppress("UNCHECKED_CAST")
        return when (event) {
            is ValueEvent.Add<*, *> ->
                ValueEvent.Add(
                    event.value as T,
                    event.addedValue as TChildren,
                    event.id,
                    path,
                    this,
                )
            is ValueEvent.Remove<*, *> ->
                ValueEvent.Remove(
                    event.value as T,
                    event.removedValue as TChildren,
                    event.id,
                    path,
                    this,
                )
            else -> super.replaceEvent(path, event)
        }
    }

    override suspend fun isValidRemoveFragment(value: T, fragment: AbsolutePathFragment): Boolean =
        innerSchema.isValidRemoveFragment(value, fragment)

    override suspend fun remove(
        path: AbsolutePath,
        value: T,
        fragment: AbsolutePathFragment,
        eventsBus: SchemaEventsBus,
    ) =
        withReplacedEvents(path, eventsBus) { replacedEventsBus ->
            innerSchema.remove(path, value, fragment, replacedEventsBus)
        }

    override fun childrenStatesContainer(): CollectionState = innerSchema.childrenStatesContainer()
}
