package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.datatypes.BigDecimal
import tech.ostack.kform.datatypes.BigInteger
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions

/** Schema representing numeric values of type [BigInteger]. */
public open class BigIntegerSchema(
    validations: Iterable<Validation<BigInteger>> = emptyList(),
    override val initialValue: BigInteger = BigInteger.ZERO,
) : AbstractSimpleSchema<BigInteger>(validations) {
    public constructor(
        vararg validations: Validation<BigInteger>,
        initialValue: BigInteger = BigInteger.ZERO,
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            BigInteger::class,
            restrictions =
                commonRestrictions(validations) + comparableBoundsRestrictions(validations),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(BigInteger.ZERO) == true

    override suspend fun fromAny(value: Any?): BigInteger =
        when (value) {
            is BigInteger -> value
            is BigDecimal -> value.toBigInteger()
            is Byte -> BigInteger.of(value.toInt())
            is Short -> BigInteger.of(value.toInt())
            is Int -> BigInteger.of(value)
            is Long -> BigInteger.of(value)
            is Float -> BigInteger.of(value.toLong())
            is Double -> BigInteger.of(value.toLong())
            is String -> BigInteger.of(value)
            else -> throw IllegalArgumentException("Cannot convert value '$value' to BigInteger.")
        }
}
