package tech.ostack.kform.schemas

import tech.ostack.kform.*

/**
 * Abstract schema representing simple (non-parent) values of type [T].
 *
 * **Note**: schemas representing non-primitive values extending this class might need to implement
 * [clone], as the default implementation might not be desirable when values are mutable.
 */
public abstract class AbstractSimpleSchema<T>(validations: Iterable<Validation<T>>) : Schema<T> {
    override val validations: List<Validation<T>> = validations.toList()

    // Default implementation which is valid for non-mutable values; simple schemas holding
    // mutable values should override this function
    override suspend fun clone(value: T): T = value

    @Suppress("UNCHECKED_CAST") public open suspend fun fromAny(value: Any?): T = clone(value as T)

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: T) -> Unit,
    ) {
        val newValue = fromAny(fromValue)
        setValue(newValue)
        eventsBus.emit(ValueEvent.Init(newValue, path, this))
    }

    override suspend fun change(
        path: AbsolutePath,
        value: T,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: T) -> Unit,
    ) {
        val newValue = fromAny(intoValue)
        if (value != newValue) {
            setValue(newValue)
            eventsBus.emit(ValueEvent.Change(value, newValue, path, this))
        } else {
            setValue(value)
        }
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: T,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (value: T) -> Unit,
    ) {
        removeValue(value)
        eventsBus.emit(ValueEvent.Destroy(value, path, this))
    }
}
