package tech.ostack.kform.internal.actions

import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.launch
import tech.ostack.kform.*
import tech.ostack.kform.internal.*
import tech.ostack.kform.schemas.ComputedSchema
import tech.ostack.kform.validations.MatchesStatefulComputedValue

/** Action writing values and their state. */
internal abstract class WriteValueStateAction<T>(formManager: FormManager) :
    ValueStateAction<T>(formManager) {
    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Write),
            AccessValidationState(ActionAccessType.Write),
            AccessStatefulValidationDeferredState(ActionAccessType.Read),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )

    // Keep track of validations to invalidate, stateful validations to update, and external issues
    // to remove
    val validationsToInvalidate = mutableSetOf<ValidationToInvalidate>()
    val statefulValidationsToUpdate = mutableSetOf<StatefulValidationToUpdate>()
    val externalIssuesToRemove = mutableListOf<LocatedValidationIssue>()
    val computedValuesToUpdate = mutableSetOf<AbsolutePath>()
    val statefulComputedValuesToUpdate = mutableSetOf<StatefulComputedValueToUpdate>()

    /**
     * Runs [runWriteValueState] and finishes up by scheduling actions to invalidate validations,
     * update stateful validations, and remove external issues.
     */
    final override suspend fun runValueState(): T =
        runWriteValueState().also {
            if (statefulValidationsToUpdate.isNotEmpty()) {
                formManager.scheduleAction(
                    UpdateStatefulValidationsStateAction(formManager, statefulValidationsToUpdate)
                )
            }
            if (validationsToInvalidate.isNotEmpty()) {
                formManager.scheduleAction(
                    InvalidateValidationsAction(formManager, validationsToInvalidate)
                )
            }
            if (statefulComputedValuesToUpdate.isNotEmpty()) {
                formManager.scheduleAction(
                    UpdateStatefulComputedValuesStateAction(
                        formManager,
                        statefulComputedValuesToUpdate,
                    )
                )
            }
            if (computedValuesToUpdate.isNotEmpty()) {
                formManager.scheduleAction(ComputeValuesAction(formManager, computedValuesToUpdate))
            }
            if (externalIssuesToRemove.isNotEmpty()) {
                formManager.scheduleAction(
                    RemoveDependingExternalIssuesAction(formManager, externalIssuesToRemove)
                )
            }
        }

    /** Run function that should be implemented by the concrete action. */
    abstract suspend fun runWriteValueState(): T

    /**
     * Runs [fn] with an events bus to be passed to schemas, where each event emitted to it is
     * appropriately handled. This is used by [SetAction] and [RemoveAction] to handle events
     * emitted by schemas. Returns the result of [fn] after all events have been handled.
     */
    protected suspend fun <TResult> withSchemaEventsBus(
        fn: suspend (eventsChannel: SchemaEventsBus) -> TResult
    ): TResult = fn(SchemaEventsCallbackBus { event -> handleSchemaEvent(event) })

    /** Handles an event emitted by a schema. */
    private suspend fun handleSchemaEvent(event: ValueEvent<*>) {
        FormManager.logger.trace { "Handling schema event: $event" }
        @Suppress("UNCHECKED_CAST")
        when (event) {
            is ValueEvent.Init<*> -> handleSchemaInitEvent(event as ValueEvent.Init<Any?>)
            is ValueEvent.Change<*> -> handleSchemaChangeEvent(event as ValueEvent.Change<Any?>)
            is ValueEvent.Destroy<*> -> handleSchemaDestroyAction(event as ValueEvent.Destroy<Any?>)
            is ValueEvent.Add<*, *> -> handleSchemaAddAction(event as ValueEvent.Add<Any?, Any?>)
            is ValueEvent.Remove<*, *> ->
                handleSchemaRemoveAction(event as ValueEvent.Remove<Any?, Any?>)
        }
    }

    protected suspend fun setOnParent(
        parentPath: AbsolutePath?,
        fragment: AbsolutePathFragment?,
        toSet: Any?,
    ) {
        // Set root value
        if (parentPath == null) {
            @Suppress("UNCHECKED_CAST")
            return withSchemaEventsBus { eventsBus ->
                val formSchema = formManager.formSchema as Schema<Any?>
                val valueToSet = if (toSet === INITIAL_VALUE) formSchema.initialValue else toSet
                if (formManager.formValue === UNINITIALIZED) {
                    formSchema.init(AbsolutePath.ROOT, valueToSet, eventsBus) {
                        formManager.formValue = it
                    }
                } else {
                    formSchema.change(
                        AbsolutePath.ROOT,
                        formManager.formValue,
                        valueToSet,
                        eventsBus,
                    ) {
                        formManager.formValue = it
                    }
                }
            }
        }

        fragment ?: error("Fragment cannot be null when [parentPath] is not null.")
        valueInfo(parentPath).collect { parentInfo ->
            withSchemaEventsBus { eventsBus -> setOnParent(parentInfo, fragment, toSet, eventsBus) }
        }
    }

    /**
     * Returns the state of the value with the provided [path] and [schema], creating it and its
     * parent states when necessary.
     */
    private fun initState(path: AbsolutePath, schema: Schema<*>): StateImpl {
        var parentState: ParentState? = null
        var fragment: AbsolutePathFragment.Id? = null
        var state =
            if (path == AbsolutePath.ROOT) formManager.formState
            else {
                parentState =
                    path.parent().let {
                        initState(it, schemaInfo(it).single().schema) as ParentState
                    }
                fragment = path.lastFragment as AbsolutePathFragment.Id
                parentState.childrenStates(path, fragment).singleOrNull()?.state
            }
                as StateImpl?
        if (state == null) {
            val nValidations = schema.validations.size
            val nStatefulValidations =
                schema.validations.count { validation ->
                    validation is StatefulValidation<*, *> ||
                        validation is MatchesStatefulComputedValue<*, *>
                }
            state =
                when (schema) {
                    is CollectionSchema<*, *> ->
                        CollectionStateImpl(
                            schema.childrenStatesContainer(),
                            nValidations,
                            nStatefulValidations,
                        )
                    is ParentSchema<*> ->
                        ParentStateImpl(
                            schema.childrenStatesContainer(),
                            nValidations,
                            nStatefulValidations,
                        )
                    else -> StateImpl(nValidations, nStatefulValidations)
                }
            if (path == AbsolutePath.ROOT) {
                formManager.formState = state
            } else {
                parentState!!.setState(fragment!!, state)
            }
        }
        return state
    }

    /**
     * Destroys the state of the value with the provided [path] and all children states. This
     * function **can** be called with the path of a value whose state has already been destroyed.
     */
    private fun destroyState(path: AbsolutePath) {
        // Keep track of how many values were destroyed that were displaying local errors/warnings
        var destroyedDisplayingLocalError = 0
        var destroyedDisplayingLocalWarning = 0

        for (info in stateInfo(path + AbsolutePathFragment.RecursiveWildcard)) {
            val state = info.state as StateImpl? ?: continue

            when (state.localDisplayStatus()) {
                DisplayStatus.Error -> ++destroyedDisplayingLocalError
                DisplayStatus.Warning -> ++destroyedDisplayingLocalWarning
                else -> {}
            }

            // Destroy state (cancels deferred validation states)
            state.destroy()

            // Destroy stateful validation states (if the validation state was in the process of
            // being updated, then the above `state.destroy()` will cause the validation state to be
            // destroyed by the update state action, otherwise we destroy it here)
            var statefulValidationIndex = 0
            for (validation in info.schema.validations) {
                if (validation is StatefulValidation<*, *>) {
                    val deferredState =
                        state.getStatefulValidationDeferredState(statefulValidationIndex++)
                            ?: continue
                    @Suppress("UNCHECKED_CAST") (validation as StatefulValidation<Any?, Any?>)
                    formManager.scope.launch(CoroutineName("Destroy stateful validation state")) {
                        try {
                            // This can throw due to the above `cancel` or due to an error while
                            // updating the state, in which case we don't need to do anything as
                            // explained above; otherwise, we destroy the validation state
                            val validationState = deferredState.await()

                            FormManager.logger.trace {
                                "At '$path': Destroying validation '$validation' state: " +
                                    "$validationState"
                            }
                            validation.destroyState(validationState)
                        } catch (_: Throwable) {}
                    }
                }
            }

            // Destroy stateful computed value state, similarly to above
            if (
                info.schema is ComputedSchema &&
                    info.schema.computedValue is StatefulComputedValue<*, *>
            ) {
                @Suppress("UNCHECKED_CAST")
                val computedValue = info.schema.computedValue as StatefulComputedValue<Any?, Any?>
                val deferredState = state.statefulComputedValueDeferredState ?: continue
                formManager.scope.launch(CoroutineName("Destroy stateful computed value state")) {
                    try {
                        // This can throw due to the above `cancel` or due to an error while
                        // updating the state, in which case we don't need to do anything as
                        // explained above; otherwise, we destroy the computed value state
                        val computedValueState = deferredState.await()

                        FormManager.logger.trace {
                            "At '$path': Destroying computed value '${computedValue}' state: " +
                                "$computedValueState"
                        }
                        computedValue.destroyState(computedValueState)
                    } catch (_: Throwable) {}
                }
            }
        }

        // Update "descendants displaying issues" of ancestors
        if (
            (destroyedDisplayingLocalError != 0 || destroyedDisplayingLocalWarning != 0) &&
                path != AbsolutePath.ROOT
        ) {
            updateDescendantsDisplayingIssues(
                descendantsDisplayingIssuesToUpdate,
                path.parent(),
                -destroyedDisplayingLocalError,
                -destroyedDisplayingLocalWarning,
            )
        }
    }

    /** Removes all cached/external issues at [path]. */
    private suspend fun invalidateLocalValidations(path: AbsolutePath) {
        val (state, schema) = stateInfo(path).single()
        state as StateImpl

        val wasValidated =
            state.validationStatus == ValidationStatus.Validated ||
                state.validationStatus == ValidationStatus.ValidatedExceptionally
        val oldLocalDisplayStatus = state.localDisplayStatus()
        val oldDisplayStatus = state.displayStatus()

        state.removeCachedIssues()
        val removedExternalIssues = state.removeExternalIssues()
        if (removedExternalIssues.isNotEmpty()) {
            formManager.externalIssuesDependencies.removeDependenciesOfExternalIssues(
                removedExternalIssues
            )
        }

        // Update validation state as needed
        if (wasValidated) {
            state.validationStatus = ValidationStatus.Unvalidated
        }
        ValidateAction.updateValidationState(
            formManager,
            schema,
            path,
            state,
            oldLocalDisplayStatus,
            oldDisplayStatus,
            descendantsDisplayingIssuesToUpdate,
            emitValidationChange = wasValidated || removedExternalIssues.isNotEmpty(),
        )
    }

    /** Updates the set of validations to invalidate with the validations that depend on [path]. */
    private fun updateValidationsToInvalidate(path: AbsolutePath) {
        for ((dependedPath, dependencyInfo) in formManager.validationDependencies.entries(path)) {
            // Depended path, with wildcards replaced by the ids of the provided [path]
            val mappedDependedPath =
                AbsolutePath(
                    dependedPath.withoutDescendants().fragments.mapIndexed { index, fragment ->
                        if (fragment is AbsolutePathFragment.Wildcard) path[index] else fragment
                    }
                )
            validationsToInvalidate +=
                ValidationToInvalidate(
                    mappedDependedPath.resolve(dependencyInfo.path),
                    dependencyInfo.validation,
                    dependencyInfo.validationIndex,
                )
        }
    }

    /**
     * Updates the set of computed values to update with the computed values that depend on the
     * [event]'s path.
     */
    private fun updateComputedValuesToCompute(event: ValueEvent<*>) {
        if (event.schema is ComputedSchema) {
            computedValuesToUpdate += event.path
        }
        for ((dependedPath, dependencyInfo) in
            formManager.computedValueDependencies.entries(event.path)) {
            // Depended path, with wildcards replaced by the ids of the provided [path]
            val mappedDependedPath =
                AbsolutePath(
                    dependedPath.withoutDescendants().fragments.mapIndexed { index, fragment ->
                        if (fragment is AbsolutePathFragment.Wildcard) event.path[index]
                        else fragment
                    }
                )
            computedValuesToUpdate += mappedDependedPath.resolve(dependencyInfo.path)
        }
    }

    /**
     * Updates the list of external issues to remove with the external issues depending on (but not
     * set on) [path].
     */
    private suspend fun updateExternalIssuesToRemove(path: AbsolutePath) {
        externalIssuesToRemove +=
            formManager.externalIssuesDependencies.getAndRemoveExternalIssuesDependentOnPath(path)
    }

    /**
     * Updates the set of stateful validations that need updating with the stateful validations
     * observing the given [path].
     */
    private fun updateStatefulValidationsToUpdate(event: ValueEvent<*>) {
        for ((
            observingPath, validation, validationIndex, statefulValidationIndex, toObserveIndex) in
            formManager.validationObservedDependencies[event.path]) {
            val resolvedObservingPath = event.path.resolve(observingPath)

            // Mustn't update state with `init`/`destroy` events of the value being validated
            if (
                (event.path != resolvedObservingPath ||
                    (event !is ValueEvent.Init && event !is ValueEvent.Destroy))
            ) {
                @Suppress("UNCHECKED_CAST")
                statefulValidationsToUpdate +=
                    StatefulValidationToUpdate(
                        resolvedObservingPath,
                        validation as StatefulValidation<Any?, Any?>,
                        validationIndex,
                        statefulValidationIndex,
                        toObserveIndex,
                        event as ValueEvent<Any?>,
                    )
            }
        }
    }

    /**
     * Updates the set of stateful computed values that need updating with the stateful computed
     * values observing the given [path].
     */
    private fun updateStatefulComputedValuesToUpdate(event: ValueEvent<*>) {
        for ((observingPath, computedValue, toObserveIndex) in
            formManager.computedValueObservedDependencies[event.path]) {
            val resolvedObservingPath = event.path.resolve(observingPath)

            // Mustn't update state with `init`/`destroy` events of the value being computed
            if (
                (event.path != resolvedObservingPath ||
                    (event !is ValueEvent.Init && event !is ValueEvent.Destroy))
            ) {
                @Suppress("UNCHECKED_CAST")
                statefulComputedValuesToUpdate +=
                    StatefulComputedValueToUpdate(
                        resolvedObservingPath,
                        computedValue as StatefulComputedValue<Any?, Any?>,
                        toObserveIndex,
                        event as ValueEvent<Any?>,
                    )
            }
        }
    }

    /** Updates all dependents related to the provided [event]. */
    private suspend fun updateDependents(event: ValueEvent<Any?>) {
        updateValidationsToInvalidate(event.path)
        updateStatefulValidationsToUpdate(event)
        updateExternalIssuesToRemove(event.path)
        updateComputedValuesToCompute(event)
        updateStatefulComputedValuesToUpdate(event)
    }

    /** Handles an "init [event]" emitted by a schema. */
    private suspend fun handleSchemaInitEvent(event: ValueEvent.Init<Any?>) {
        // Initialise state and possibly parent states
        initState(event.path, event.schema)

        updateDependents(event)
        formManager.eventsBus.emit(event)
    }

    /** Handles a "change [event]" emitted by a schema. */
    private suspend fun handleSchemaChangeEvent(event: ValueEvent.Change<Any?>) {
        invalidateLocalValidations(event.path)
        updateDependents(event)
        formManager.eventsBus.emit(event)
    }

    /** Handles a "destroy [event]" emitted by a schema. */
    private suspend fun handleSchemaDestroyAction(event: ValueEvent.Destroy<Any?>) {
        // Destroy state of destroyed value (state may already have been destroyed by a remove)
        destroyState(event.path)

        // Set state to `null` in parent state when the child state exists
        val parentState = stateInfo(event.path.parent()).singleOrNull()?.state as ParentState?
        val id = event.path.lastFragment as AbsolutePathFragment.Id
        if (parentState != null && (parentState !is CollectionState || parentState.hasState(id))) {
            parentState.setState(id, null)
        }

        updateDependents(event)
        formManager.eventsBus.emit(event)
    }

    /** Handles an "add [event]" emitted by a schema. */
    private suspend fun handleSchemaAddAction(event: ValueEvent.Add<Any?, Any?>) {
        invalidateLocalValidations(event.path)
        updateDependents(event)
        formManager.eventsBus.emit(event)
    }

    /** Handles a "remove [event]" emitted by a schema. */
    private suspend fun handleSchemaRemoveAction(event: ValueEvent.Remove<Any?, Any?>) {
        // Destroy state of removed value and remove its state from parent state
        val parentState = stateInfo(event.path).single().state as CollectionState
        destroyState(event.path + event.id)
        parentState.removeState(event.id)

        invalidateLocalValidations(event.path)
        updateDependents(event)
        formManager.eventsBus.emit(event)
    }
}
