package tech.ostack.kform.internal.actions

import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.FormManager
import tech.ostack.kform.internal.AccessExternalContext
import tech.ostack.kform.internal.ActionAccessType

/**
 * Action that writes the external context with name [externalContextName] available to validations.
 */
internal abstract class WriteExternalContextAction<T>(formManager: FormManager) :
    FormManagerAction<T>(formManager) {
    abstract val externalContextName: String

    override val accesses = listOf(AccessExternalContext(ActionAccessType.Write))
    override val accessedPaths: List<AbsolutePath> = emptyList()

    // FIXME: Cannot add function below due to the external contexts map not being thread-safe
    // override fun overridesConflictingAction(action: Action<*>): Boolean =
    //     action is WriteExternalContextAction
    //         && action.externalContextName == externalContextName

    /**
     * Invalidate all validations and recompute all computed values depending on the external
     * context, as well as remove external issues depending on it.
     */
    protected suspend fun updateDependents() {
        // Invalidate depending validations
        val toInvalidate = mutableSetOf<ValidationToInvalidate>()
        for ((path, validation, validationIndex) in
            formManager.externalContextValidationDependencies[externalContextName] ?: emptyList()) {
            toInvalidate += ValidationToInvalidate(path, validation, validationIndex)
        }
        if (toInvalidate.isNotEmpty()) {
            formManager.scheduleAction(InvalidateValidationsAction(formManager, toInvalidate))
        }

        // Remove depending external issues
        val dependingExternalIssues =
            formManager.externalIssuesDependencies
                .getAndRemoveExternalIssuesDependentOnExternalContext(externalContextName)
        if (dependingExternalIssues.isNotEmpty()) {
            formManager.scheduleAction(
                RemoveDependingExternalIssuesAction(formManager, dependingExternalIssues)
            )
        }

        // Recompute computed values
        val dependingComputedValues =
            formManager.externalContextComputedValueDependencies[externalContextName] ?: emptyList()
        if (dependingComputedValues.isNotEmpty()) {
            formManager.scheduleAction(
                ComputeValuesAction(formManager, dependingComputedValues.map { it.path })
            )
        }
    }
}
