package tech.ostack.kform.internal.actions

import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/** Action performed on values and/or their state. */
internal abstract class ValueStateAction<T>(formManager: FormManager) :
    FormManagerAction<T>(formManager) {
    // Map of descendants displaying issues to update
    val descendantsDisplayingIssuesToUpdate =
        mutableMapOf<AbsolutePath, DescendantsDisplayingIssuesToUpdate>()

    final override suspend fun run(): T =
        runValueState().also {
            if (descendantsDisplayingIssuesToUpdate.isNotEmpty()) {
                formManager.scheduleAction(
                    UpdateDescendantsDisplayingIssuesAction(
                        formManager,
                        descendantsDisplayingIssuesToUpdate.values,
                    )
                )
            }
        }

    /** Run function that should be implemented by the concrete action. */
    abstract suspend fun runValueState(): T

    /**
     * List of the parents of [path]. When marking values as dirty/touched or changing the number of
     * touched descendants with issues, all parent values must also be marked as dirty/touched; this
     * function helps with locking them.
     */
    fun parentPaths(path: AbsolutePath): List<AbsolutePath> =
        if (path.isRoot) emptyList()
        else
            buildList {
                var currentPath = path.parent()
                add(currentPath)
                while (!currentPath.isRoot) {
                    currentPath = currentPath.parent()
                    add(currentPath)
                }
            }

    fun schemaInfo(path: AbsolutePath) = schemaInfoImpl(formManager.formSchema, path)

    fun valueInfo(path: AbsolutePath) =
        @Suppress("UNCHECKED_CAST")
        valueInfoImpl(formManager.formSchema as Schema<Any?>, formManager.formValue, path)

    fun stateInfo(path: AbsolutePath) =
        @Suppress("UNCHECKED_CAST")
        stateInfoImpl(formManager.formSchema as Schema<Any?>, formManager.formState!!, path)

    fun valueStateInfo(path: AbsolutePath) =
        @Suppress("UNCHECKED_CAST")
        valueStateInfoImpl(
            formManager.formSchema as Schema<Any?>,
            formManager.formValue,
            formManager.formState!!,
            path,
        )

    suspend fun dependenciesInfo(
        path: AbsolutePath,
        dependencies: Map<String, DependencyInfo>,
    ): Map<String, ValueInfo<*>?> =
        @Suppress("UNCHECKED_CAST")
        computationDependenciesInfo(
            formManager.formSchema as Schema<Any?>,
            formManager.formValue,
            path,
            dependencies,
        )

    companion object {
        fun updateDescendantsDisplayingIssues(
            descendantsDisplayingIssuesToUpdate:
                MutableMap<AbsolutePath, DescendantsDisplayingIssuesToUpdate>,
            path: AbsolutePath,
            displayingErrors: Int = 0,
            displayingWarnings: Int = 0,
        ) {
            val toUpdate =
                descendantsDisplayingIssuesToUpdate.getOrPut(path) {
                    DescendantsDisplayingIssuesToUpdate(path)
                }
            toUpdate.displayingErrors += displayingErrors
            toUpdate.displayingWarnings += displayingWarnings
        }

        fun updateDescendantsDisplayingIssues(
            descendantsDisplayingIssuesToUpdate:
                MutableMap<AbsolutePath, DescendantsDisplayingIssuesToUpdate>,
            path: AbsolutePath,
            severity: ValidationIssueSeverity,
            amount: Int,
        ) =
            updateDescendantsDisplayingIssues(
                descendantsDisplayingIssuesToUpdate,
                path,
                if (severity == ValidationIssueSeverity.Error) amount else 0,
                if (severity == ValidationIssueSeverity.Warning) amount else 0,
            )
    }
}
