package tech.ostack.kform.internal.actions

import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.launch
import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/** Information about a stateful validation needing updating. */
internal class StatefulValidationToUpdate(
    val path: AbsolutePath,
    val validation: StatefulValidation<Any?, Any?>,
    val validationIndex: Int,
    val statefulValidationIndex: Int,
    val observerIndex: Int,
    val event: ValueEvent<Any?>,
) {
    override fun equals(other: Any?) =
        when {
            this === other -> true
            other !is StatefulValidationToUpdate -> false
            else ->
                path == other.path &&
                    validationIndex == other.validationIndex &&
                    event == other.event
        }

    override fun hashCode(): Int {
        var result = path.hashCode()
        result = 31 * result + validationIndex
        result = 31 * result + event.hashCode()
        return result
    }
}

/**
 * Action that updates the state of all provided stateful validations. This action runs at normal
 * priority since the updates are done in new coroutines that this action does not wait for.
 */
internal class UpdateStatefulValidationsStateAction(
    formManager: FormManager,
    private val toUpdate: Set<StatefulValidationToUpdate>,
) : ValueStateAction<Unit>(formManager) {
    override fun toString(): String =
        "UpdateStatefulValidationsState(${toUpdate.joinToString {
            "${it.validation}@${it.path} ← ${it.event}"
        }})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Read),
            AccessStatefulValidationDeferredState(ActionAccessType.Write),
        )
    override val accessedPaths = toUpdate.map { it.path }.toHashSet()

    override suspend fun runValueState() =
        toUpdate.forEach { validationInfo ->
            stateInfo(validationInfo.path).forEach { info ->
                updateStatefulValidationState(
                    info.state as StateImpl?,
                    info.schema,
                    info.path,
                    validationInfo.validation,
                    validationInfo.validationIndex,
                    validationInfo.statefulValidationIndex,
                    validationInfo.observerIndex,
                    validationInfo.event,
                )
            }
        }

    private fun updateStatefulValidationState(
        state: StateImpl?,
        schema: Schema<*>,
        path: AbsolutePath,
        validation: StatefulValidation<Any?, Any?>,
        validationIndex: Int,
        statefulValidationIndex: Int,
        observerIndex: Int,
        event: ValueEvent<Any?>,
    ) {
        state ?: return

        // Do nothing if the validation state hasn't yet been initialised
        val oldDeferredValidationState =
            state.getStatefulValidationDeferredState(statefulValidationIndex) ?: return
        val newDeferredValidationState = CompletableDeferred<Any?>(formManager.supervisorJob)
        state.setStatefulValidationDeferredState(
            newDeferredValidationState,
            statefulValidationIndex,
        )

        // Propagate cancellation downwards (when a value is destroyed before an up-to-date
        // validation state has been computed)
        newDeferredValidationState.invokeOnCompletion { oldDeferredValidationState.cancel() }

        // By the time we update the validation state, a `ValidateAction` may be waiting for the new
        // state; this only happens when the validation wasn't previously (successfully) validated.
        // This means that we cannot always issue a `RemoveCachedIssuesAction` since we may be
        // removing issues that were cached **after** the validation state was set. Thus, we check
        // whether the validation was previously validated before issuing a
        // `RemoveCachedIssuesAction`.
        val validationWasValidated =
            state.getCachedIssues(validationIndex).let { cachedIssues ->
                cachedIssues != null && !cachedIssues.any { it is ValidationFailure }
            }

        formManager.scope.launch(CoroutineName("Update stateful validation state")) {
            FormManager.logger.trace {
                "At '$path': Handling validation '$validation' event: $event"
            }
            val oldValidationState: Any?
            try {
                oldValidationState = oldDeferredValidationState.await()
            } catch (ex: Throwable) {
                // Propagate exception upwards, so that exceptions on previous validation
                // states always propagate to the latest one
                newDeferredValidationState.completeExceptionally(ex)
                return@launch
            }

            try {
                val newValidationState =
                    validation.observers[observerIndex].updateState(oldValidationState, event)
                if (newDeferredValidationState.complete(newValidationState)) {
                    if (oldValidationState != newValidationState) {
                        if (validationWasValidated) {
                            formManager.scheduleAction(
                                RemoveCachedIssuesAction(
                                    formManager,
                                    path,
                                    schema,
                                    state,
                                    validation,
                                    validationIndex,
                                )
                            )
                        }
                    }
                    FormManager.logger.debug {
                        "At '$path': Updated validation '$validation' state: " +
                            "$oldValidationState -> $newValidationState"
                    }
                } else {
                    // Deferred was cancelled before we completed, so new state is useless
                    FormManager.logger.trace {
                        "At '$path': Destroying validation '$validation' new state: " +
                            "$newValidationState"
                    }
                    validation.destroyState(newValidationState)
                }
            } catch (ex: Throwable) {
                // An error occurred during [updateState]
                FormManager.logger.debug {
                    "At '$path': Failed to update validation '$validation' state ($ex)"
                }
                newDeferredValidationState.completeExceptionally(ex)
            }

            // Always destroy old state
            FormManager.logger.trace {
                "At '$path': Destroying validation '$validation' old state: " +
                    "$oldValidationState"
            }
            validation.destroyState(oldValidationState)
        }
    }
}
