package tech.ostack.kform.internal.actions

import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.launch
import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/** Information about a stateful computed value needing updating. */
internal class StatefulComputedValueToUpdate(
    val path: AbsolutePath,
    val computedValue: StatefulComputedValue<Any?, Any?>,
    val observerIndex: Int,
    val event: ValueEvent<Any?>,
) {
    override fun equals(other: Any?) =
        when {
            this === other -> true
            other !is StatefulComputedValueToUpdate -> false
            else -> path == other.path && event == other.event
        }

    override fun hashCode(): Int {
        var result = path.hashCode()
        result = 31 * result + event.hashCode()
        return result
    }
}

/**
 * Action that updates the state of all provided stateful computed values. This action runs at
 * normal priority since the updates are done in new coroutines that this action does not wait for.
 */
internal class UpdateStatefulComputedValuesStateAction(
    formManager: FormManager,
    private val toUpdate: Set<StatefulComputedValueToUpdate>,
) : ValueStateAction<Unit>(formManager) {
    override fun toString(): String =
        "UpdateStatefulComputedValuesState(${toUpdate.joinToString {
            "${it.computedValue}@${it.path} ← ${it.event}"
        }})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessStatefulComputedValueDeferredState(ActionAccessType.Write),
        )
    override val accessedPaths = toUpdate.map { it.path }.toHashSet()

    override suspend fun runValueState() =
        toUpdate.forEach { validationInfo ->
            stateInfo(validationInfo.path).forEach { info ->
                updateStatefulComputedValueState(
                    info.state as StateImpl?,
                    info.path,
                    validationInfo.computedValue,
                    validationInfo.observerIndex,
                    validationInfo.event,
                )
            }
        }

    private fun updateStatefulComputedValueState(
        state: StateImpl?,
        path: AbsolutePath,
        computedValue: StatefulComputedValue<Any?, Any?>,
        observerIndex: Int,
        event: ValueEvent<Any?>,
    ) {
        state ?: return

        // Do nothing if the computed value state hasn't yet been initialised
        val oldDeferredComputedValueState = state.statefulComputedValueDeferredState ?: return
        val newDeferredComputedValueState = CompletableDeferred<Any?>(formManager.supervisorJob)
        state.statefulComputedValueDeferredState = newDeferredComputedValueState

        // Propagate cancellation downwards (when a value is destroyed before an up-to-date
        // computed value state has been computed)
        newDeferredComputedValueState.invokeOnCompletion { oldDeferredComputedValueState.cancel() }

        formManager.scope.launch(CoroutineName("Update stateful computed value state")) {
            FormManager.logger.trace {
                "At '$path': Handling computed value '$computedValue' event: $event"
            }
            val oldComputedValueState: Any?
            try {
                oldComputedValueState = oldDeferredComputedValueState.await()
            } catch (ex: Throwable) {
                // Propagate exception upwards, so that exceptions on previous computed value
                // states always propagate to the latest one
                newDeferredComputedValueState.completeExceptionally(ex)
                return@launch
            }

            try {
                val newComputedValueState =
                    computedValue.observers[observerIndex].updateState(oldComputedValueState, event)
                if (newDeferredComputedValueState.complete(newComputedValueState)) {
                    if (oldComputedValueState != newComputedValueState) {
                        formManager.scheduleAction(ComputeValuesAction(formManager, listOf(path)))
                    }
                    FormManager.logger.debug {
                        "At '$path': Updated computed value '$computedValue' state: " +
                            "$oldComputedValueState -> $newComputedValueState"
                    }
                } else {
                    // Deferred was cancelled before we completed, so new state is useless
                    FormManager.logger.trace {
                        "At '$path': Destroying computed value '$computedValue' new state: " +
                            "$newComputedValueState"
                    }
                    computedValue.destroyState(newComputedValueState)
                }
            } catch (ex: Throwable) {
                // An error occurred during [updateState]
                FormManager.logger.debug {
                    "At '$path': Failed to update computed value '$computedValue' state ($ex)"
                }
                newDeferredComputedValueState.completeExceptionally(ex)
            }

            // Always destroy old state
            FormManager.logger.trace {
                "At '$path': Destroying computed value '$computedValue' old state: " +
                    "$oldComputedValueState"
            }
            computedValue.destroyState(oldComputedValueState)
        }
    }
}
