package tech.ostack.kform.internal.actions

import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.FormManager
import tech.ostack.kform.StateEvent
import tech.ostack.kform.internal.*

/** Information about descendants displaying issues to update. */
internal class DescendantsDisplayingIssuesToUpdate(val path: AbsolutePath) {
    var displayingErrors: Int = 0
    var displayingWarnings: Int = 0
}

/**
 * Action that updates the number of descendants displaying issues of each severity at each provided
 * path, as well as their parents.
 */
internal class UpdateDescendantsDisplayingIssuesAction(
    formManager: FormManager,
    private val toUpdate: Iterable<DescendantsDisplayingIssuesToUpdate>,
) : ValueStateAction<Unit>(formManager) {
    private fun signedNumber(n: Int) = if (n < 0) "$n" else "+$n"

    override fun toString() =
        "UpdateDesdendantsDisplayingIssues(${toUpdate.joinToString {
            "${it.path} ${
                if (it.displayingErrors == 0) "" else "${signedNumber(it.displayingErrors)}E"
            }${
                if (it.displayingWarnings == 0) "" else "${signedNumber(it.displayingWarnings)}W"
            }"
        }})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Read),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Write),
        )
    override val accessedPaths = toUpdate.flatMap { parentPaths(it.path) + it.path }.toSet()

    override suspend fun runValueState() =
        toUpdate.forEach {
            updateDescendantsDisplayingIssues(it.path, it.displayingErrors, it.displayingWarnings)
        }

    private tailrec suspend fun updateDescendantsDisplayingIssues(
        path: AbsolutePath,
        displayingErrors: Int,
        displayingWarnings: Int,
    ) {
        val info = stateInfo(path).single()
        val state = info.state as ParentStateImpl

        val oldDisplayStatus = state.displayStatus()
        state.descendantsDisplayingErrors += displayingErrors
        state.descendantsDisplayingWarnings += displayingWarnings
        val newDisplayStatus = state.displayStatus()

        if (oldDisplayStatus != newDisplayStatus) {
            formManager.eventsBus.emit(
                StateEvent.DisplayChange(newDisplayStatus, path, info.schema)
            )
        }
        if (path != AbsolutePath.ROOT) {
            updateDescendantsDisplayingIssues(path.parent(), displayingErrors, displayingWarnings)
        }
    }
}
