package tech.ostack.kform.internal.actions

import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/**
 * Action that sets all values whose path matches [path] as untouched, as well as their children.
 */
internal class SetUntouchedAction(formManager: FormManager, private val path: AbsolutePath) :
    ValueStateAction<Unit>(formManager) {
    override fun toString() = "SetUntouched($path)"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessIsTouched(ActionAccessType.Write),
            AccessValidationState(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )
    override val accessedPaths = listOf(path + AbsolutePathFragment.RecursiveWildcard)

    override suspend fun runValueState() =
        stateInfo(path).forEach { info ->
            setUntouched(info.path, info.schema, info.state as StateImpl?)
        }

    private suspend fun setUntouched(path: AbsolutePath, schema: Schema<*>, state: StateImpl?) {
        if (state != null && state.isTouched) {
            val oldDisplayStatus = state.displayStatus()
            state.isTouched = false
            formManager.eventsBus.emit(StateEvent.TouchedChange(false, path, schema))

            if (oldDisplayStatus != DisplayStatus.Valid) {
                formManager.eventsBus.emit(
                    StateEvent.DisplayChange(DisplayStatus.Valid, path, schema)
                )
            }

            if (path != AbsolutePath.ROOT) {
                val dominantSeverity = state.dominantIssueSeverity()
                if (dominantSeverity != null) {
                    updateDescendantsDisplayingIssues(
                        descendantsDisplayingIssuesToUpdate,
                        path.parent(),
                        dominantSeverity,
                        -1,
                    )
                }
            }

            if (state is ParentState) {
                for (childInfo in state.childrenStates(path, AbsolutePathFragment.Wildcard)) {
                    if (childInfo.state != null) {
                        setUntouched(childInfo.path, childInfo.schema, childInfo.state as StateImpl)
                    }
                }
            }
        }
    }
}
