package tech.ostack.kform.internal.actions

import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.FormManager
import tech.ostack.kform.Schema
import tech.ostack.kform.internal.*

/**
 * Action that clears all external issues at paths matching [path] and with code [code] from the
 * form manager.
 */
internal class RemoveExternalIssuesAction(
    formManager: FormManager,
    private val path: AbsolutePath,
    private val code: String?,
) : ValueStateAction<Unit>(formManager) {
    override fun toString() =
        "RemoveExternalIssues($path${if (code == null) "" else ", code=$code"})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Write),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )
    override val accessedPaths = listOf(path)

    override suspend fun runValueState() =
        stateInfo(path).forEach { info ->
            removeExternalIssues(info.state as StateImpl?, info.schema, info.path)
        }

    private suspend fun removeExternalIssues(
        state: StateImpl?,
        schema: Schema<*>,
        path: AbsolutePath,
    ) {
        state ?: return

        // Old statuses
        val oldLocalDisplayStatus = state.localDisplayStatus()
        val oldDisplayStatus = state.displayStatus()

        val removedIssues = state.removeExternalIssues(code)
        if (removedIssues.isNotEmpty()) {
            formManager.externalIssuesDependencies.removeDependenciesOfExternalIssues(removedIssues)

            // Update validation state with the new external issues
            ValidateAction.updateValidationState(
                formManager,
                schema,
                path,
                state,
                oldLocalDisplayStatus,
                oldDisplayStatus,
                descendantsDisplayingIssuesToUpdate,
            )
        }
    }
}
