package tech.ostack.kform.internal

import tech.ostack.kform.*
import tech.ostack.kform.collections.PathMultimap
import tech.ostack.kform.collections.mutablePathMultimapOf
import tech.ostack.kform.collections.set

/**
 * Multimap mapping paths of values being depended upon to information about what validations depend
 * on them.
 *
 * Example: if a schema with path `"/x/y/z"` has a single validation `v` depending on the path
 * `"../a"`, then there will be an entry in this multimap mapping `"/x/y/a"` to
 * `ValidationDependencyInfo("../z", v, 0)`.
 */
internal typealias ValidationDependencies = PathMultimap<ValidationDependencyInfo>

/**
 * Multimap mapping paths of values being observed to information about what validations are
 * observing them.
 *
 * Example: if a schema with path `"/x/y/z"` has a single validation `v` (that validation being
 * stateful) needing to observe a single path `"../a"`, then there will be an entry in this multimap
 * mapping `"/x/y/a"` to `ObservedValidationDependencyInfo("../z", v, 0, 0, 0)`.
 */
internal typealias ObservedValidationDependencies = PathMultimap<ObservedValidationDependencyInfo>

/** Map with information on the list of validations depending on each external context. */
internal typealias ExternalContextValidationDependencies =
    Map<String, List<ExternalContextValidationDependantInfo>>

/** Information about a validation depending on a certain path. */
internal data class ValidationDependencyInfo(
    val path: Path,
    val validation: Validation<*>,
    val validationIndex: Int,
) {
    override fun toString() = "$validation@$path"
}

/** Information about a stateful validation observing a certain path. */
internal data class ObservedValidationDependencyInfo(
    val path: Path,
    val validation: StatefulValidation<*, *>,
    val validationIndex: Int,
    val statefulValidationIndex: Int,
    val toObserveIndex: Int,
) {
    override fun toString() = "$validation@$path"
}

/** Information about a validation that depends on external context. */
internal data class ExternalContextValidationDependantInfo(
    val path: AbsolutePath,
    val validation: Validation<*>,
    val validationIndex: Int,
) {
    override fun toString() = "$validation@$path"
}

/**
 * Builds and returns the multimap with information on path dependencies for all validations in
 * [formSchema].
 */
internal fun buildValidationDependencies(formSchema: Schema<*>): ValidationDependencies {
    val pathDependencies = mutablePathMultimapOf<ValidationDependencyInfo>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        for ((validationIndex, validation) in schema.validations.withIndex()) {
            // Have all descendants depend on [validation] when [dependsOnDescendants] is set, e.g.,
            // if the validation path is `/a/b`, then this results in `/a/b/*/** -> ..`
            if (validation.dependsOnDescendants) {
                pathDependencies[path + Path.DESCENDANTS] =
                    ValidationDependencyInfo(Path.PARENT, validation, validationIndex)
            }

            validateComputation(formSchema, path, validation)
            for (dependencyInfo in validation.dependencies.values) {
                val resolvedDep = path.resolve(dependencyInfo.path)
                pathDependencies[resolvedDep] =
                    ValidationDependencyInfo(
                        path.relativeTo(resolvedDep.withoutDescendants()),
                        validation,
                        validationIndex,
                    )
            }
        }
    }
    return pathDependencies
}

/**
 * Builds and returns the multimap with information on observed path dependencies for all stateful
 * validations in [formSchema].
 */
internal fun buildObservedValidationDependencies(
    formSchema: Schema<*>
): ObservedValidationDependencies {
    val observedPathDependencies = mutablePathMultimapOf<ObservedValidationDependencyInfo>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        var statefulValidationIndex = 0
        for ((validationIndex, validation) in schema.validations.withIndex()) {
            if (validation is StatefulValidation<*, *>) {
                // Paths to observe must be contained by one of these paths
                val toObserveContainers =
                    listOf(path + AbsolutePathFragment.RecursiveWildcard) +
                        validation.dependencies.values.map {
                            path.resolve(it.path) + AbsolutePathFragment.RecursiveWildcard
                        }

                for ((observedIndex, observer) in validation.observers.withIndex()) {
                    val resolvedToObserve = path.resolve(observer.toObserve)

                    if (
                        toObserveContainers.all { resolvedToObserve !in it } ||
                            !schemaInfoImpl(formSchema, resolvedToObserve).any()
                    ) {
                        throw InvalidPathToObserveException(path, validation, observer.toObserve)
                    }

                    for (resolvedInfo in schemaInfoImpl(formSchema, resolvedToObserve)) {
                        observedPathDependencies[resolvedInfo.queriedPath] =
                            ObservedValidationDependencyInfo(
                                path.relativeTo(resolvedInfo.queriedPath),
                                validation,
                                validationIndex,
                                statefulValidationIndex,
                                observedIndex,
                            )
                    }
                }
                ++statefulValidationIndex
            }
        }
    }
    return observedPathDependencies
}

/** Builds and returns the map with information on validations depending on external contexts. */
internal fun buildExternalContextValidationDependencies(
    formSchema: Schema<*>
): ExternalContextValidationDependencies {
    val externalContextDependencies =
        hashMapOf<String, MutableList<ExternalContextValidationDependantInfo>>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        for ((validationIndex, validation) in schema.validations.withIndex()) {
            for (externalContextName in validation.externalContextDependencies) {
                externalContextDependencies.getOrPut(externalContextName) { mutableListOf() } +=
                    ExternalContextValidationDependantInfo(path, validation, validationIndex)
            }
        }
    }
    return externalContextDependencies
}
