package tech.ostack.kform.internal

import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.emitAll
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.*

/**
 * Returns information about the inner schemas of [formSchema] matching [path] (recursive
 * implementation).
 *
 * @param curInfo Information of schema being currently analysed.
 * @param i Index of the fragment of the path being analysed.
 */
internal fun <T> schemaInfoImpl(
    formSchema: Schema<T>,
    path: AbsolutePath,
    curInfo: SchemaInfo<*> = SchemaInfo(formSchema, AbsolutePath.ROOT),
    i: Int = 0,
): Sequence<SchemaInfo<*>> = sequence {
    if (i == path.size) {
        yield(curInfo)
    } else {
        val (curSchema, curPath, curQueriedPath) = curInfo
        val fragment = path[i]
        if (fragment is AbsolutePathFragment.RecursiveWildcard) {
            yieldAll(schemaInfoImpl(formSchema, path, curInfo, i + 1))
            if (curSchema is ParentSchema<*>) {
                for (childSchemaInfo in
                    curSchema.childrenSchemas(
                        curPath,
                        curQueriedPath,
                        AbsolutePathFragment.Wildcard,
                    )) {
                    yieldAll(schemaInfoImpl(formSchema, path, childSchemaInfo, i))
                }
            }
        } else if (
            curSchema is ParentSchema<*> &&
                (fragment == AbsolutePathFragment.Wildcard ||
                    curSchema.isValidChildSchemaFragment(fragment))
        ) {
            for (childSchemaInfo in curSchema.childrenSchemas(curPath, curQueriedPath, fragment)) {
                yieldAll(schemaInfoImpl(formSchema, path, childSchemaInfo, i + 1))
            }
        }
    }
}

/**
 * Returns a flow of information about the parts of a value [formValue] with schema [formSchema]
 * matching [path] (recursive implementation).
 *
 * @param curInfo Information of value being currently analysed.
 * @param i Index of the fragment of the path being analysed.
 */
internal fun <T> valueInfoImpl(
    formSchema: Schema<T>,
    formValue: T,
    path: AbsolutePath,
    curInfo: ValueInfo<*> = ValueInfo(formValue, formSchema, AbsolutePath.ROOT, AbsolutePath.ROOT),
    i: Int = 0,
): Flow<ValueInfo<*>> = flow {
    if (i == path.size) {
        emit(curInfo)
    } else {
        val (curValue, curSchema, curPath, curSchemaPath) = curInfo
        val fragment = path[i]
        if (fragment is AbsolutePathFragment.RecursiveWildcard) {
            emitAll(valueInfoImpl(formSchema, formValue, path, curInfo, i + 1))
            if (curSchema is ParentSchema<*>) {
                @Suppress("UNCHECKED_CAST") (curSchema as ParentSchema<Any?>)
                curSchema
                    .children(curPath, curSchemaPath, curValue, AbsolutePathFragment.Wildcard)
                    .collect { childInfo: ValueInfo<*> ->
                        @Suppress("UNCHECKED_CAST") (childInfo as ValueInfo<Any?>)
                        emitAll(valueInfoImpl(formSchema, formValue, path, childInfo, i))
                    }
            }
        } else if (curSchema is ParentSchema<*>) {
            @Suppress("UNCHECKED_CAST") (curSchema as ParentSchema<Any?>)
            if (
                fragment == AbsolutePathFragment.Wildcard ||
                    curSchema.isValidChildFragment(curValue, fragment)
            ) {
                curSchema.children(curPath, curSchemaPath, curValue, fragment).collect {
                    childInfo: ValueInfo<*> ->
                    @Suppress("UNCHECKED_CAST") (childInfo as ValueInfo<Any?>)
                    emitAll(valueInfoImpl(formSchema, formValue, path, childInfo, i + 1))
                }
            }
        }
    }
}

/**
 * Returns a sequence of information about the parts of [formState] of schema [formSchema] matching
 * [path] (recursive implementation).
 *
 * @param curInfo Information of value being currently analysed.
 * @param i Index of the fragment of the path being analysed.
 */
internal fun <T> stateInfoImpl(
    formSchema: Schema<T>,
    formState: State,
    path: AbsolutePath,
    curInfo: StateInfo<*> = StateInfo(formState, formSchema, AbsolutePath.ROOT),
    i: Int = 0,
): Sequence<StateInfo<*>> = sequence {
    if (i == path.size) {
        yield(curInfo)
    } else {
        val (curState, _, curPath) = curInfo
        val fragment = path[i]
        if (fragment is AbsolutePathFragment.RecursiveWildcard) {
            yieldAll(stateInfoImpl(formSchema, formState, path, curInfo, i + 1))
            if (curState is ParentState) {
                for (childInfo in curState.childrenStates(curPath, AbsolutePathFragment.Wildcard)) {
                    yieldAll(stateInfoImpl(formSchema, formState, path, childInfo, i))
                }
            }
        } else if (curState is ParentState) {
            for (childInfo in curState.childrenStates(curPath, fragment)) {
                yieldAll(stateInfoImpl(formSchema, formState, path, childInfo, i + 1))
            }
        }
    }
}

/**
 * Returns a flow of information about the parts of a value [formValue] with schema [formSchema]
 * matching [path] (recursive implementation).
 *
 * @param curInfo Information of value being currently analysed.
 * @param i Index of the fragment of the path being analysed.
 */
@Suppress("UNCHECKED_CAST")
internal fun <T> valueStateInfoImpl(
    formSchema: Schema<T>,
    formValue: T,
    formState: State,
    path: AbsolutePath,
    curInfo: ValueStateInfo<*> =
        ValueStateInfo(formValue, formState, formSchema, AbsolutePath.ROOT, AbsolutePath.ROOT),
    i: Int = 0,
): Flow<ValueStateInfo<*>> = flow {
    if (i == path.size) {
        emit(curInfo)
    } else {
        val (curValue, curState, curSchema, curPath, curSchemaPath) = curInfo
        val fragment = path[i]
        if (fragment is AbsolutePathFragment.RecursiveWildcard) {
            emitAll(valueStateInfoImpl(formSchema, formValue, formState, path, curInfo, i + 1))
            if (curSchema is ParentSchema<*>) {
                curSchema as ParentSchema<Any?>
                curSchema
                    .children(curPath, curSchemaPath, curValue, AbsolutePathFragment.Wildcard)
                    .collect { childValueInfo: ValueInfo<*> ->
                        childValueInfo as ValueInfo<Any?>
                        val childStateInfo =
                            (curState as ParentState)
                                .childrenStates(curPath, childValueInfo.path.lastFragment!!)
                                .single()
                        emitAll(
                            valueStateInfoImpl(
                                formSchema,
                                formValue,
                                formState,
                                path,
                                ValueStateInfo(
                                    childValueInfo.value,
                                    childStateInfo.state,
                                    childValueInfo.schema,
                                    childValueInfo.path,
                                    childValueInfo.schemaPath,
                                ),
                                i,
                            )
                        )
                    }
            }
        } else if (curSchema is ParentSchema<*>) {
            curSchema as ParentSchema<Any?>
            if (
                fragment == AbsolutePathFragment.Wildcard ||
                    curSchema.isValidChildFragment(curValue, fragment)
            ) {
                curSchema.children(curPath, curSchemaPath, curValue, fragment).collect {
                    childValueInfo: ValueInfo<*> ->
                    childValueInfo as ValueInfo<Any?>
                    val childStateInfo =
                        (curState as ParentState)
                            .childrenStates(curPath, childValueInfo.path.lastFragment!!)
                            .single()
                    emitAll(
                        valueStateInfoImpl(
                            formSchema,
                            formValue,
                            formState,
                            path,
                            ValueStateInfo(
                                childValueInfo.value,
                                childStateInfo.state,
                                childValueInfo.schema,
                                childValueInfo.path,
                                childValueInfo.schemaPath,
                            ),
                            i + 1,
                        )
                    )
                }
            }
        }
    }
}
