package tech.ostack.kform.internal

import tech.ostack.kform.*
import tech.ostack.kform.collections.PathMultimap
import tech.ostack.kform.collections.mutablePathMultimapOf
import tech.ostack.kform.collections.set
import tech.ostack.kform.schemas.ComputedSchema

/**
 * Multimap mapping paths of values being depended upon to information about what computed values
 * depend on them.
 *
 * Example: if the computed value `c` of a computed schema with path `"/x/y/z"` depends on path
 * `"../a"`, then there will be an entry in this multimap mapping `"/x/y/a"` to
 * `ComputedValueDependencyInfo("../z", c)`.
 */
internal typealias ComputedValueDependencies = PathMultimap<ComputedValueDependencyInfo>

/**
 * Multimap mapping paths of values being observed to information about what computed values are
 * observing them.
 *
 * Example: if the stateful computed value `c` of a computed schema with path `"/x/y/z"` needs to
 * observe a single path `"../a"`, then there will be an entry in this multimap mapping `"/x/y/a"`
 * to `ObservedComputedValueDependencyInfo("../z", c)`.
 */
internal typealias ObservedComputedValueDependencies =
    PathMultimap<ObservedComputedValueDependencyInfo>

/** Map with information on the list of computedValues depending on each external context. */
internal typealias ExternalContextComputedValueDependencies =
    Map<String, List<ExternalContextComputedValueDependantInfo>>

/** Information about a computed value depending on a certain path. */
internal data class ComputedValueDependencyInfo(
    val path: Path,
    val computedValue: ComputedValue<*>,
) {
    override fun toString() = "$computedValue@$path"
}

/** Information about a stateful computed value observing a certain path. */
internal data class ObservedComputedValueDependencyInfo(
    val path: Path,
    val computedValue: StatefulComputedValue<*, *>,
    val toObserveIndex: Int,
) {
    override fun toString() = "$computedValue@$path"
}

/** Information about a computed value that depends on external context. */
internal data class ExternalContextComputedValueDependantInfo(
    val path: AbsolutePath,
    val computedValue: ComputedValue<*>,
) {
    override fun toString() = "$computedValue@$path"
}

/**
 * Builds and returns the multimap with information on path dependencies for all computedValues in
 * [formSchema].
 */
internal fun buildComputedValueDependencies(formSchema: Schema<*>): ComputedValueDependencies {
    val pathDependencies = mutablePathMultimapOf<ComputedValueDependencyInfo>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        if (schema is ComputedSchema) {
            val computedValue = schema.computedValue

            validateComputation(formSchema, path, computedValue)
            for (dependencyInfo in computedValue.dependencies.values) {
                val resolvedDep = path.resolve(dependencyInfo.path)
                pathDependencies[resolvedDep] =
                    ComputedValueDependencyInfo(
                        path.relativeTo(resolvedDep.withoutDescendants()),
                        computedValue,
                    )
            }
        }
    }
    return pathDependencies
}

/**
 * Builds and returns the multimap with information on observed path dependencies for all stateful
 * computed values in [formSchema].
 */
internal fun buildObservedComputedValueDependencies(
    formSchema: Schema<*>
): ObservedComputedValueDependencies {
    val observedPathDependencies = mutablePathMultimapOf<ObservedComputedValueDependencyInfo>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        if (schema is ComputedSchema && schema.computedValue is StatefulComputedValue<*, *>) {
            val computedValue = schema.computedValue as StatefulComputedValue<*, *>

            // Paths to observe must be contained by one of these paths
            val toObserveContainers =
                listOf(path + AbsolutePathFragment.RecursiveWildcard) +
                    computedValue.dependencies.values.map {
                        path.resolve(it.path) + AbsolutePathFragment.RecursiveWildcard
                    }

            for ((observedIndex, observer) in computedValue.observers.withIndex()) {
                val resolvedToObserve = path.resolve(observer.toObserve)

                if (
                    toObserveContainers.all { resolvedToObserve !in it } ||
                        !schemaInfoImpl(formSchema, resolvedToObserve).any()
                ) {
                    throw InvalidPathToObserveException(path, computedValue, observer.toObserve)
                }

                for (resolvedInfo in schemaInfoImpl(formSchema, resolvedToObserve)) {
                    observedPathDependencies[resolvedInfo.queriedPath] =
                        ObservedComputedValueDependencyInfo(
                            path.relativeTo(resolvedInfo.queriedPath),
                            computedValue,
                            observedIndex,
                        )
                }
            }
        }
    }
    return observedPathDependencies
}

/**
 * Builds and returns the map with information on computed values depending on external contexts.
 */
internal fun buildExternalContextComputedValueDependencies(
    formSchema: Schema<*>
): ExternalContextComputedValueDependencies {
    val externalContextDependencies =
        hashMapOf<String, MutableList<ExternalContextComputedValueDependantInfo>>()
    for ((schema, path) in schemaInfoImpl(formSchema, AbsolutePath.MATCH_ALL)) {
        if (schema is ComputedSchema) {
            val computedValue = schema.computedValue

            for (externalContextName in computedValue.externalContextDependencies) {
                externalContextDependencies.getOrPut(externalContextName) { mutableListOf() } +=
                    ExternalContextComputedValueDependantInfo(path, computedValue)
            }
        }
    }
    return externalContextDependencies
}
