package tech.ostack.kform.internal

import kotlin.collections.component1
import kotlin.collections.component2
import kotlin.collections.iterator
import kotlinx.coroutines.flow.singleOrNull
import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.AbsolutePathFragment
import tech.ostack.kform.Computation
import tech.ostack.kform.DependencyInfo
import tech.ostack.kform.InvalidDependencyPathException
import tech.ostack.kform.InvalidDependencyTypeException
import tech.ostack.kform.Schema
import tech.ostack.kform.ValueInfo

/** Returns the values of the provided [dependencies] as needed by a computation at path [path]. */
internal suspend fun <T, S : Schema<T>> computationDependenciesInfo(
    formSchema: S,
    formValue: T,
    path: AbsolutePath,
    dependencies: Map<String, DependencyInfo>,
): Map<String, ValueInfo<*>?> =
    dependencies.mapValues { (_, dep) ->
        // Single value matching the dependency or `null` if no such value exists
        valueInfoImpl(formSchema, formValue, path.resolve(dep.path).withoutDescendants())
            .singleOrNull()
    }

/** Validates a given [computation] at path [path]. */
internal fun validateComputation(
    formSchema: Schema<*>,
    path: AbsolutePath,
    computation: Computation,
) {
    for ((dependencyKey, dependencyInfo) in computation.dependencies) {
        val (dependencyPath, dependencyType) = dependencyInfo
        val dependencyPathWithoutDescendants = dependencyPath.withoutDescendants()

        val schemaInfo =
            schemaInfoImpl(formSchema, path.resolve(dependencyPathWithoutDescendants))
                .singleOrNull()
        if (
            dependencyPathWithoutDescendants.fragments.any { fragment ->
                fragment is AbsolutePathFragment.Wildcard ||
                    fragment is AbsolutePathFragment.RecursiveWildcard ||
                    fragment is AbsolutePathFragment.CollectionEnd
            } || schemaInfo == null
        ) {
            throw InvalidDependencyPathException(path, computation, dependencyKey, dependencyPath)
        }
        if (!schemaInfo.schema.assignableTo(dependencyType)) {
            throw InvalidDependencyTypeException(
                path,
                computation,
                schemaInfo.schema.typeInfo.toString(),
                dependencyKey,
                dependencyType,
            )
        }
    }
}
