@file:JvmName("Files")

package tech.ostack.kform.datatypes

import kotlin.io.encoding.Base64
import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder

/**
 * Representation of a file to be used in a form.
 *
 * @property name Name of the file.
 * @property data File content.
 * @property type File MIME type.
 */
@Serializable
public data class File
@JvmOverloads
public constructor(
    public val name: String = DEFAULT_FILE_NAME,
    public val data: ByteArray = byteArrayOf(),
    public val type: String? = null,
) {
    @JvmOverloads
    public constructor(data: ByteArray, type: String? = null) : this(DEFAULT_FILE_NAME, data, type)

    init {
        require(name.isNotEmpty()) { "File name cannot be empty." }
    }

    /** Size of the file's data, in bytes. */
    public val size: Int
        get() = data.size

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is File -> false
            name != other.name -> false
            !data.contentEquals(other.data) -> false
            type != other.type -> false
            else -> true
        }

    override fun hashCode(): Int {
        var result = name.hashCode()
        result = 31 * result + data.contentHashCode()
        result = 31 * result + (type?.hashCode() ?: 0)
        return result
    }

    override fun toString(): String =
        "File(name=$name, data=ByteArray(size=${data.size})${
            if (type == null) "" else ", type=$type"
        })"

    public companion object {
        /** Default file name. */
        public const val DEFAULT_FILE_NAME: String = "unnamed_file"

        /** Empty placeholder file instance. */
        internal val EMPTY_PLACEHOLDER_FILE = File("empty_file")
    }

    /**
     * Serializer used to serialize the file's [data] as base 64. Useful when using formats such as
     * JSON which do not support binary data.
     *
     * Example usage:
     * ```kotlin
     * @Serializable
     * data class Person(
     *   val name: String,
     *   val address: String,
     *   @Serializable(with = File.Base64Serializer::class)
     *   val proofOfAddress: File?
     * )
     * ```
     */
    public object Base64Serializer : KSerializer<File> {
        /** Helper class used to serialise a file with its data in base 64. */
        @Serializable
        @SerialName("tech.ostack.kform.datatypes.File")
        private class Surrogate(val name: String, val data: String = "", val type: String? = null)

        override val descriptor: SerialDescriptor = Surrogate.serializer().descriptor

        override fun serialize(encoder: Encoder, value: File): Unit =
            encoder.encodeSerializableValue(
                Surrogate.serializer(),
                Surrogate(value.name, Base64.encode(value.data), value.type),
            )

        override fun deserialize(decoder: Decoder): File {
            val surrogate = decoder.decodeSerializableValue(Surrogate.serializer())
            return File(surrogate.name, Base64.decode(surrogate.data), surrogate.type)
        }
    }
}

/** Function returning an empty, placeholder file. */
public fun emptyPlaceholderFile(): File = File.EMPTY_PLACEHOLDER_FILE
