package tech.ostack.kform.computedvalues

import kotlin.jvm.JvmName
import kotlin.jvm.JvmStatic
import kotlin.reflect.KMutableProperty1
import tech.ostack.kform.AbsolutePathFragment
import tech.ostack.kform.Computation
import tech.ostack.kform.ComputedValue
import tech.ostack.kform.ComputedValueContext
import tech.ostack.kform.Path
import tech.ostack.kform.datatypes.BigDecimal
import tech.ostack.kform.datatypes.BigInteger
import tech.ostack.kform.datatypes.Table
import tech.ostack.kform.schemas.BigDecimalSchema
import tech.ostack.kform.valueInfo

/**
 * Computed value which computes the sum of all values in a column of a table at [tablePath]. Values
 * of type [Int], [Long], [Double], [BigInteger], and [BigDecimal] are supported.
 *
 * Specify the column over which to compute the sum, relative to a row of the table, via
 * [columnPath].
 *
 * Column values should be (possibly) nullable numeric values.
 *
 * Example usage:
 * ```kotlin
 * ComputedSchema(SumOf("../table", Row::column)) {
 *   LongSchema()
 * }
 * ```
 */
public interface SumOf<T> : Computation {
    /** Path of the table over which to compute the sum. */
    public val tablePath: Path

    /** Path of the table column over which to compute the sum, relative to a row of the table. */
    public val columnPath: Path

    public companion object {
        @Suppress("UNCHECKED_CAST")
        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T> invoke(
            tablePath: Path,
            columnPath: Path,
        ): ComputedValue<T> =
            when (T::class) {
                Int::class -> IntSumOf(tablePath, columnPath)
                Long::class -> LongSumOf(tablePath, columnPath)
                Double::class -> DoubleSumOf(tablePath, columnPath)
                BigInteger::class -> BigIntegerSumOf(tablePath, columnPath)
                BigDecimal::class -> BigDecimalSumOf(tablePath, columnPath)
                else ->
                    error(
                        "The SumOf computed value can only be used on schemas of type Int, Long, Double, " +
                            "BigInteger, or BigDecimal."
                    )
            }
                as ComputedValue<T>

        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T> invoke(
            tablePath: Path,
            columnProperty: KMutableProperty1<*, *>,
        ): ComputedValue<T> = SumOf<T>(tablePath, Path(columnProperty.name))

        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T> invoke(
            tablePath: String,
            columnPath: String,
        ): ComputedValue<T> = SumOf<T>(Path(tablePath), Path(columnPath))

        @JvmStatic
        @JvmName("create")
        public inline operator fun <reified T> invoke(
            tablePath: String,
            columnProperty: KMutableProperty1<*, *>,
        ): ComputedValue<T> = SumOf<T>(Path(tablePath), Path(columnProperty.name))
    }
}

/** [SumOf] implementation for an [Int] schema. */
@PublishedApi
internal open class IntSumOf(override val tablePath: Path, override val columnPath: Path) :
    SumOf<Int>, ComputedValue<Int>() {
    private val ComputedValueContext.table by
        dependency<Table<Any>>(tablePath + AbsolutePathFragment.RecursiveWildcard)

    override fun toString(): String = "SumOf($tablePath, $columnPath)"

    override suspend fun ComputedValueContext.compute(): Int {
        var sum = 0
        valueInfo(dependencySchema("table"), table, Path.CHILDREN + columnPath).collect { info ->
            sum += (info.value as Number?)?.toInt() ?: 0
        }
        return sum
    }
}

/** [SumOf] implementation for a [Long] schema. */
@PublishedApi
internal open class LongSumOf(override val tablePath: Path, override val columnPath: Path) :
    SumOf<Long>, ComputedValue<Long>() {
    private val ComputedValueContext.table by
        dependency<Table<Any>>(tablePath + AbsolutePathFragment.RecursiveWildcard)

    override fun toString(): String = "SumOf($tablePath, $columnPath)"

    override suspend fun ComputedValueContext.compute(): Long {
        var sum = 0L
        valueInfo(dependencySchema("table"), table, Path.CHILDREN + columnPath).collect { info ->
            sum += (info.value as Number?)?.toLong() ?: 0L
        }
        return sum
    }
}

/** [SumOf] implementation for a [Double] schema. */
@PublishedApi
internal open class DoubleSumOf(override val tablePath: Path, override val columnPath: Path) :
    SumOf<Double>, ComputedValue<Double>() {
    private val ComputedValueContext.table by
        dependency<Table<Any>>(tablePath + AbsolutePathFragment.RecursiveWildcard)

    override fun toString(): String = "SumOf($tablePath, $columnPath)"

    override suspend fun ComputedValueContext.compute(): Double {
        var sum = 0.0
        valueInfo(dependencySchema("table"), table, Path.CHILDREN + columnPath).collect { info ->
            sum += (info.value as Number?)?.toDouble() ?: 0.0
        }
        return sum
    }
}

/** [SumOf] implementation for a [BigInteger] schema. */
@PublishedApi
internal open class BigIntegerSumOf(override val tablePath: Path, override val columnPath: Path) :
    SumOf<BigInteger>, ComputedValue<BigInteger>() {
    private val ComputedValueContext.table by
        dependency<Table<Any>>(tablePath + AbsolutePathFragment.RecursiveWildcard)

    override fun toString(): String = "SumOf($tablePath, $columnPath)"

    override suspend fun ComputedValueContext.compute(): BigInteger {
        var sum = BigInteger.ZERO
        valueInfo(dependencySchema("table"), table, Path.CHILDREN + columnPath).collect { info ->
            sum += info.value.toBigInteger()
        }
        return sum
    }

    private fun Any?.toBigInteger(): BigInteger =
        when (this) {
            null -> BigInteger.ZERO
            is BigInteger -> this
            is BigDecimal -> this.toBigInteger()
            is Byte -> BigInteger.of(this.toInt())
            is Short -> BigInteger.of(this.toInt())
            is Int -> BigInteger.of(this)
            is Long -> BigInteger.of(this)
            is Float -> BigInteger.of(this.toLong())
            is Double -> BigInteger.of(this.toLong())
            else -> throw IllegalArgumentException("Cannot convert value '$this' to BigInteger.")
        }
}

/** [SumOf] implementation for a [BigDecimal] schema. */
@PublishedApi
internal open class BigDecimalSumOf(override val tablePath: Path, override val columnPath: Path) :
    SumOf<BigDecimal>, ComputedValue<BigDecimal>() {
    private val ComputedValueContext.table by
        dependency<Table<Any>>(tablePath + AbsolutePathFragment.RecursiveWildcard)

    override fun toString(): String = "SumOf($tablePath, $columnPath)"

    override suspend fun ComputedValueContext.compute(): BigDecimal {
        var sum = BigDecimalSchema.zeroOfPreferredScale(schema.validations)
        valueInfo(dependencySchema("table"), table, Path.CHILDREN + columnPath).collect { info ->
            sum += info.value.toBigDecimal()
        }
        return sum
    }

    private fun Any?.toBigDecimal(): BigDecimal =
        when (this) {
            null -> BigDecimal.ZERO
            is BigDecimal -> this
            is BigInteger -> BigDecimal.of(this)
            is Byte -> BigDecimal.of(this.toInt())
            is Short -> BigDecimal.of(this.toInt())
            is Int -> BigDecimal.of(this)
            is Long -> BigDecimal.of(this)
            is Float -> BigDecimal.of(this)
            is Double -> BigDecimal.of(this)
            else -> error("Cannot convert value '$this' to BigDecimal.")
        }
}
