package tech.ostack.kform

import kotlin.js.JsName
import kotlin.jvm.JvmName
import kotlin.properties.PropertyDelegateProvider
import kotlin.properties.ReadOnlyProperty
import kotlin.reflect.typeOf
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.emitAll
import kotlinx.coroutines.flow.flow

/**
 * Context provided to a validation when running it, contains the values of the validation's
 * dependencies at the time the validation was executed.
 */
public open class ValidationContext(
    private val value: Any?,
    schema: Schema<*>,
    /** Path of the value being validated. */
    path: AbsolutePath,
    /** Path of the schema of the value being validated. */
    schemaPath: AbsolutePath,
    /**
     * Value information for each dependency of the validation. Mapping of keys as declared in the
     * [validation dependencies][Validation.dependencies] to their respective value information.
     */
    dependenciesInfo: DependenciesValueInfo,
    externalContexts: ExternalContexts,
) : ComputationContext(schema, path, schemaPath, dependenciesInfo, externalContexts) {
    /** Value being validated. */
    @Suppress("UNCHECKED_CAST") public fun <T> value(): T = value as T
}

/**
 * Validation for values of type [T].
 *
 * Validations may depend on other values. These dependencies may be defined via the [dependency]
 * function, by providing the path of the dependency relative to the value being validated.
 *
 * Example validation that emits an error when the integer being validated is odd if the value of a
 * dependency `allowOdd` is `false`:
 * ```kotlin
 * object DisallowOdd : Validation<Int>() {
 *     private val ValidationContext.allowOdd: Boolean by dependency("../allowOdd")
 *
 *     ValidationContext.validate() = flow {
 *         if (!allowOdd && value % 2 != 0) {
 *             emit(ValidationError("oddNotAllowed"))
 *         }
 *     }
 * }
 * ```
 */
@JsName("ValidationKt")
public abstract class Validation<in T> : Computation {
    /** Schema of the value being validated. */
    protected inline val ValidationContext.schema: Schema<@UnsafeVariance T>
        get() = schema()

    /** Value being validated. */
    protected inline val ValidationContext.value: @UnsafeVariance T
        get() = value()

    /**
     * Dependencies of the validation. Mapping of keys to the paths this validation depends on. Keys
     * can be used within a [ValidationContext] to access the value of the dependencies.
     *
     * Validation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the validation should be reevaluated by the [form manager][FormManager] whenever a child
     * value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    public override val dependencies: Map<String, DependencyInfo> = hashMapOf()

    /**
     * Whether the [form manager][FormManager] should reevaluate this validation whenever a
     * descendant of the value being validated changes. This is `false` by default.
     *
     * Conceptually, when `false`, it is as if the validation has an implicit dependency on the path
     * `"."` (itself). When `true`, this dependency becomes `"./∗∗"` (itself and all of its
     * descendants).
     */
    public open val dependsOnDescendants: Boolean = false

    /**
     * Set of external context dependencies of the validation.
     *
     * Each entry of this set represents the name of the external context being depended upon.
     */
    public override val externalContextDependencies: Set<String> = hashSetOf()

    /**
     * Declares a dependency to [path], accessible in the validation's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    @JvmName("reifiedAddDependency")
    protected inline fun <reified TDependency> addDependency(dependencyKey: String, path: Path) {
        (dependencies as MutableMap<String, DependencyInfo>)[dependencyKey] =
            DependencyInfo(path, typeOf<TDependency>())
    }

    /**
     * Declares a dependency to [path], accessible in the validation's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addDependency(dependencyKey: String, path: Path): Unit =
        addDependency<Any?>(dependencyKey, path)

    /**
     * Declares a dependency to [path], accessible in the validation's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    @JvmName("reifiedAddDependency")
    protected inline fun <reified TDependency> addDependency(
        dependencyKey: String,
        path: String,
    ): Unit = addDependency<TDependency>(dependencyKey, Path(path))

    /**
     * Declares a dependency to [path], accessible in the validation's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addDependency(dependencyKey: String, path: String): Unit =
        addDependency<Any?>(dependencyKey, path)

    /**
     * Declares an external context dependency to [externalContextName].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addExternalContextDependency(externalContextName: String) {
        (externalContextDependencies as MutableSet<String>).add(externalContextName)
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ValidationContext].
     *
     * If the dependency could not be found in the form during the execution of the validation, then
     * accessing this dependency will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.dependencyKey: Type? by dependencyOrNull(path)
     * ```
     *
     * Validation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the validation should be reevaluated by the [form manager][FormManager] whenever a child
     * value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    protected inline fun <reified TDependency> dependencyOrNull(
        path: Path,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TDependency?>,
    > = PropertyDelegateProvider { validation, property ->
        validation.addDependency<TDependency>(dependencyKey ?: property.name, path)
        // Return a delegate to the dependency value within a [ValidationContext]
        ReadOnlyProperty { ctx, prop -> ctx.dependencyOrNull(dependencyKey ?: prop.name) }
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ValidationContext].
     *
     * If the dependency could not be found in the form during the execution of the validation, then
     * accessing this dependency will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.dependencyKey: Type? by dependencyOrNull(path)
     * ```
     *
     * Validation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the validation should be reevaluated by the [form manager][FormManager] whenever a child
     * value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    protected inline fun <reified TDependency> dependencyOrNull(
        path: String,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TDependency?>,
    > = dependencyOrNull(Path(path), dependencyKey)

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ValidationContext].
     *
     * If the dependency could not be found in the form during the execution of the validation, then
     * accessing this dependency will throw [DependencyNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.dependencyKey: Type by dependency(path)
     * ```
     *
     * Validation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the validation should be reevaluated by the [form manager][FormManager] whenever a child
     * value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    protected inline fun <reified TDependency> dependency(
        path: Path,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TDependency>,
    > = PropertyDelegateProvider { validation, property ->
        validation.addDependency<TDependency>(dependencyKey ?: property.name, path)
        // Return a delegate to the dependency value within a [ValidationContext]
        ReadOnlyProperty { ctx, prop -> ctx.dependency(dependencyKey ?: prop.name) }
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ValidationContext].
     *
     * If the dependency could not be found in the form during the execution of the validation, then
     * accessing this dependency will throw [DependencyNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.dependencyKey: Type by dependency(path)
     * ```
     *
     * Validation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the validation should be reevaluated by the [form manager][FormManager] whenever a child
     * value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    protected inline fun <reified TDependency> dependency(
        path: String,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TDependency>,
    > = dependency(Path(path), dependencyKey)

    /**
     * Function used to declare a dependency to an external context and delegate access to its value
     * within a [ValidationContext].
     *
     * If the external context could not be found during the execution of the validation, then
     * accessing this external context will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.externalContextName: Type? by externalContextOrNull()
     * ```
     */
    protected fun <TContext> externalContextOrNull(
        externalContextName: String? = null
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TContext?>,
    > = PropertyDelegateProvider { validation, property ->
        validation.addExternalContextDependency(externalContextName ?: property.name)
        // Return a delegate to the external context dependency value within a [ValidationContext]
        ReadOnlyProperty { ctx, prop ->
            ctx.externalContextOrNull(externalContextName ?: prop.name)
        }
    }

    /**
     * Function used to declare a dependency to an external context and delegate access to its value
     * within a [ValidationContext].
     *
     * If the external context could not be found during the execution of the validation, then
     * accessing this external context will throw [ExternalContextNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ValidationContext.externalContextName: Type by externalContext()
     * ```
     */
    protected fun <TContext> externalContext(
        externalContextName: String? = null
    ): PropertyDelegateProvider<
        Validation<@UnsafeVariance T>,
        ReadOnlyProperty<ValidationContext, TContext>,
    > = PropertyDelegateProvider { validation, property ->
        validation.addExternalContextDependency(externalContextName ?: property.name)
        // Return a delegate to the external context dependency value within a [ValidationContext]
        ReadOnlyProperty { ctx, prop -> ctx.externalContext(externalContextName ?: prop.name) }
    }

    /**
     * Runs the validation within a [ValidationContext] containing the [value] being validated and
     * the value of all declared dependencies. Returns a flow over all found issues.
     */
    public abstract fun ValidationContext.validate(): Flow<ValidationIssue>

    // Default [toString] implementation
    public override fun toString(): String = this::class.simpleName ?: super.toString()
}

/**
 * Validation for values of type [T] where the [form manager][FormManager] maintains a validation
 * state of type [TState] for each managed value.
 *
 * Stateful validations are useful when validations require an expensive computation over data and
 * if it is possible to save the result of such expensive computation and tweak it as data changes
 * instead of running the expensive computation all over again.
 *
 * As an example, imagine that we have a list of people and that we want to validate that the
 * average age of all people isn't over a certain age. Instead of iterating over the whole list
 * every time a person is added or removed, we can use a stateful validation to save, as state, the
 * sum of all ages, and simply tweak this sum as people are added or removed. Having access to the
 * sum of all ages as state allows us to implement the validation function with a complexity of O(1)
 * as opposed to O(N).
 *
 * The following snippet implements the example above of making sure that the average age of all
 * people doesn't surpass a dependency `maxAvgAge`:
 * ```kotlin
 * object AvgAgeNotOverMax : StatefulValidation<List<Person>, Int>() {
 *     private val ValidationContext.maxAvgAge: Int by dependency("../maxAvgAge")
 *
 *     override suspend fun ValidationContext.initState(value: List<Person>): Int =
 *         value.fold(0) { sum, person -> sum + person.age }
 *
 *     private val ageObserver by observe<Int>("∗/age") { agesSum, event ->
 *         when (event) {
 *             is ValueEvent.Init<Int> -> agesSum + event.newValue
 *             is ValueEvent.Change<Int> -> agesSum + event.newValue - event.oldValue
 *             is ValueEvent.Destroy<Int> -> agesSum - event.oldValue
 *             else -> agesSum
 *         }
 *     }
 *
 *     override fun ValidationContext.validateFromState(value: List<Person>, state: Int) = flow {
 *         val avgAge = state / value.size
 *         if (avgAge > maxAvgAge) {
 *             emit(ValidationError("avgAgeOverMax"))
 *         }
 *     }
 * }
 * ```
 */
@JsName("StatefulValidationKt")
public abstract class StatefulValidation<in T, TState> :
    StatefulComputation<ValidationContext, TState>, Validation<T>() {
    /**
     * List of observers.
     *
     * Each observer contains a path to be observed and a function to update the state of the
     * validation, which will be called whenever an event with a path matching the one being
     * observed occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     */
    public override val observers: List<Observer<Any?, TState>> = mutableListOf()

    /**
     * Adds an [observer] to the path [observer.toObserve][Observer.toObserve] to update the
     * validation state via [observer.updateState][Observer.updateState] whenever an event with a
     * path matching [observer.toObserve][Observer.toObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(observer: Observer<TValue, TState>) {
        @Suppress("UNCHECKED_CAST")
        (observers as MutableList<Observer<*, TState>>) += observer
    }

    /**
     * Adds an observer to the path [pathToObserve] to update the validation state via [updateState]
     * whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(
        pathToObserve: Path,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): Unit = addObserver(Observer(pathToObserve, updateState))

    /**
     * Adds an observer to the path [pathToObserve] to update the validation state via [updateState]
     * whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(
        pathToObserve: String,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): Unit = addObserver(Path(pathToObserve), updateState)

    /**
     * Function used to observe the path [pathToObserve] and update the validation state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * It should be used as follows:
     * ```kotlin
     * private val observer by observe<Type>(path) { state, event ->
     *     when (event) {
     *         is ValueEvent.Init<Type>      -> // Compute new state
     *         is ValueEvent.Change<Type>    -> // Compute new state
     *         is ValueEvent.Destroy<Type>   -> // Compute new state
     *
     *         // These events only occur when observing a collection:
     *         is ValueEvent.Add<Type, *>    -> // Compute new state
     *         is ValueEvent.Remove<Type, *> -> // Compute new state
     *     }
     * }
     * ```
     *
     * The provided function must return the new validation state which, unless it hasn't changed,
     * should be different ([equals]-wise) from the previous validation state (as such, when using
     * an object as state, a new instance should be returned when the state is updated). If the new
     * state differs from the old one (using [equals]), the value will need to be revalidated by the
     * [form manager][FormManager] using [validateFromState].
     */
    protected fun <TValue> observe(
        pathToObserve: Path,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): PropertyDelegateProvider<
        StatefulValidation<@UnsafeVariance T, TState>,
        ReadOnlyProperty<StatefulValidation<@UnsafeVariance T, TState>, Observer<TValue, TState>>,
    > = PropertyDelegateProvider { validation, _ ->
        val observer = Observer(pathToObserve, updateState)
        validation.addObserver(observer)
        ReadOnlyProperty { _, _ -> observer }
    }

    /**
     * Function used to observe the path [pathToObserve] and update the validation state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * It should be used as follows:
     * ```kotlin
     * private val observer by observe<Type>(path) { state, event ->
     *     when (event) {
     *         is ValueEvent.Init<Type>      -> // Compute new state
     *         is ValueEvent.Change<Type>    -> // Compute new state
     *         is ValueEvent.Destroy<Type>   -> // Compute new state
     *
     *         // These events only occur when observing a collection:
     *         is ValueEvent.Add<Type, *>    -> // Compute new state
     *         is ValueEvent.Remove<Type, *> -> // Compute new state
     *     }
     * }
     * ```
     *
     * The provided function must return the new validation state which, unless it hasn't changed,
     * should be different ([equals]-wise) from the previous validation state (as such, when using
     * an object as state, a new instance should be returned when the state is updated). If the new
     * state differs from the old one (using [equals]), the value will need to be revalidated by the
     * [form manager][FormManager] using [validateFromState].
     */
    protected fun <TValue> observe(
        pathToObserve: String,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): PropertyDelegateProvider<
        StatefulValidation<@UnsafeVariance T, TState>,
        ReadOnlyProperty<StatefulValidation<@UnsafeVariance T, TState>, Observer<TValue, TState>>,
    > = observe(Path(pathToObserve), updateState)

    /**
     * Initialises and returns the validation's state, given the [value] to validate within a
     * [ValidationContext] containing the values of all declared dependencies.
     *
     * This method is called by the [form manager][FormManager] whenever a new value that needs to
     * be validated by this validation is initialised.
     */
    public abstract override suspend fun ValidationContext.initState(): TState

    /**
     * Destroys the validation [state].
     *
     * Called by the [form manager][FormManager] whenever a state is no longer needed (i.e. when a
     * newer state has been produced or when the value being validated has been destroyed).
     */
    public override suspend fun destroyState(state: TState): Unit = Unit

    /**
     * Runs the validation, given its [state], within a [ValidationContext] containing the [value]
     * being validated and the value of all declared dependencies. Returns a flow over all found
     * issues.
     *
     * This method is called by the [form manager][FormManager] instead of [validate] when
     * validating stateful validations.
     *
     * Note that the form utilities [validate][tech.ostack.kform.validate] function (and, by
     * consequence, the [form validator][FormValidator]) does **not** maintain validation states
     * and, as such, does **not** ever explicitly call any methods specific to stateful validations
     * like [validateFromState]. This is why we provide the following default implementation of
     * [validate]:
     * ```kotlin
     * override fun ValidationContext.validate(): Flow<Issue> = flow {
     *     emitAll(validateFromState(initState(value)))
     * }
     * ```
     */
    public abstract fun ValidationContext.validateFromState(state: TState): Flow<ValidationIssue>

    // Default [validate] implementation for stateful validations.
    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        emitAll(validateFromState(initState()))
    }
}
