package tech.ostack.kform

import kotlin.js.JsName

/**
 * Information about a schema of values of type [T]: the [schema] instance, and its [path].
 *
 * @property schema Schema itself.
 * @property path Absolute path of the schema within a [form validator][FormValidator] or
 *   [manager][FormManager].
 * @property queriedPath Absolute path of the queried value within a [form validator][FormValidator]
 *   or [manager][FormManager].
 *
 * This path may be either equal to the [schema path][path], or a more specific path contained by
 * it.
 */
@JsName("SchemaInfoKt")
public data class SchemaInfo<T>(
    public val schema: Schema<T>,
    public val path: AbsolutePath,
    public val queriedPath: AbsolutePath = path,
)

/**
 * Information about a value of type [T]: the [value] itself, its absolute [path], and its [schema].
 *
 * @property value Value itself.
 * @property schema Schema of the value.
 * @property path Absolute path of the value within a [form manager][FormManager] or within the
 *   value passed to a [form validator][FormValidator].
 * @property schemaPath Absolute path of the schema within a [form validator][FormValidator] or
 *   [manager][FormManager].
 */
@JsName("ValueInfoKt")
public data class ValueInfo<T>(
    public val value: T,
    public val schema: Schema<T>,
    public val path: AbsolutePath,
    public val schemaPath: AbsolutePath,
)

/**
 * Information about the state of a value.
 *
 * @property state State itself.
 * @property schema Schema of the value.
 * @property path Absolute path of the value with this state within a [form manager][FormManager].
 */
@JsName("StateInfoKt")
public data class StateInfo<T>(
    public val state: State?,
    public val schema: Schema<T>,
    public val path: AbsolutePath,
)

/**
 * Information about a value of type [T] and its state: the [value] itself, its [state], its
 * absolute [path], and its [schema].
 *
 * @property value Value itself.
 * @property state State of the value.
 * @property schema Schema of the value.
 * @property path Absolute path of the value within a [form manager][FormManager] or within the
 *   value passed to a [form validator][FormValidator].
 * @property schemaPath Absolute path of the schema within a [form validator][FormValidator] or
 *   [manager][FormManager].
 */
@JsName("ValueStateInfoKt")
internal data class ValueStateInfo<T>(
    val value: T,
    val state: State?,
    val schema: Schema<T>,
    val path: AbsolutePath,
    val schemaPath: AbsolutePath,
)

/**
 * Information about a value in the [form manager][FormManager].
 *
 * @property value Value itself.
 * @property schema Schema of the value.
 * @property path Absolute path of the value within a [form manager][FormManager].
 * @property schemaPath Absolute path of the schema within a [form manager][FormManager].
 * @property dirty Whether the value is considered dirty.
 * @property touched Whether the value is considered touched.
 * @property issues Issues associated with the value or `null` when the value has not yet been
 *   validated.
 * @property validationStatus Validation status of the value.
 * @property displayStatus Display status of the value.
 */
@JsName("InfoKt")
public data class Info<T>(
    public val value: T,
    public val schema: Schema<T>,
    public val path: AbsolutePath,
    public val schemaPath: AbsolutePath,
    public val dirty: Boolean,
    public val touched: Boolean,
    public val issues: List<ValidationIssue>,
    public val validationStatus: ValidationStatus,
    public val displayStatus: DisplayStatus,
)
