package tech.ostack.kform

import kotlinx.coroutines.flow.*

/**
 * Criteria that determines whether to run schema-internal validations and/or external validations
 * when calling [FormValidator.validate].
 *
 * The default criteria is [ExternalIfInternalValid].
 */
public enum class ValidateRunCriteria {
    /**
     * Schema-internal validations always run; external validations only run if no errors where
     * found when running the internal validations.
     *
     * This is the default behaviour.
     */
    ExternalIfInternalValid,
    /** Only the schema-internal validations are run. External validations do **not** run. */
    InternalOnly,
    /** Only the external validations are run. Schema-internal validations do **not** run. */
    ExternalOnly,
    /**
     * Both schema-internal and external validations always run. External validations run even if
     * errors are found in internal validations.
     */
    All,
}

/**
 * Class used to validate form values. Form values are of type [T] and represent the content of a
 * form with the provided form schema.
 *
 * External validations may be provided to further validate the form against validations not present
 * in the schema.
 *
 * Once instantiated with a given form schema, the validator can be used to validate values
 * according to all [validations][Validation] of said schema, as well as the validator's external
 * validations.
 */
public class FormValidator<T>(
    private val formSchema: Schema<T>,
    externalValidations: ExternalValidations = emptyMap(),
) {
    private val externalValidations: Map<PathOrString, List<Validation<*>>> =
        LinkedHashMap<PathOrString, List<Validation<*>>>(externalValidations.size).apply {
            for ((path, iterable) in externalValidations) {
                this[path.toAbsolutePath()] = iterable.toList()
            }
        }

    init {
        validateSchemaValidations(formSchema)
        validateExternalValidations(formSchema, this.externalValidations)
    }

    /**
     * Validates the parts of the form value [formValue] matching [path] against the validator's
     * schema. Returns a flow of found [validation issues][LocatedValidationIssue].
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws InvalidPathException If [path] matches no schemas.
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public fun validate(
        formValue: T,
        path: Path,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Flow<LocatedValidationIssue> = flow {
        var hasErrors = false
        if (runCriteria != ValidateRunCriteria.ExternalOnly) {
            validate(formSchema, formValue, path, externalContexts).collect { issue ->
                if (issue.severity == ValidationIssueSeverity.Error) {
                    hasErrors = true
                }
                emit(issue)
            }
        }

        if (
            externalValidations.isNotEmpty() &&
                (runCriteria == ValidateRunCriteria.ExternalOnly ||
                    runCriteria == ValidateRunCriteria.All ||
                    (runCriteria == ValidateRunCriteria.ExternalIfInternalValid && !hasErrors))
        ) {
            emitAll(
                validateExternally(
                    formSchema,
                    formValue,
                    path,
                    externalValidations,
                    externalContexts,
                )
            )
        }
    }

    /**
     * Validates the parts of the form value [formValue] matching [path] against the validator's
     * schema. Returns a flow of found [validation issues][LocatedValidationIssue].
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws InvalidPathException If [path] matches no schemas.
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public fun validate(
        formValue: T,
        path: String,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Flow<LocatedValidationIssue> =
        validate(formValue, AbsolutePath(path), externalContexts, runCriteria)

    /**
     * Validates all parts of the form value [formValue] against the validator's schema. Returns a
     * flow of found [validation issues][LocatedValidationIssue].
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public fun validate(
        formValue: T,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Flow<LocatedValidationIssue> =
        validate(formValue, AbsolutePath.MATCH_ALL, externalContexts, runCriteria)

    /**
     * Returns whether the parts of the form value [formValue] matching [path] are valid according
     * to their schemas. These parts are said to be valid if they contain no validation errors.
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws InvalidPathException If [path] matches no schemas.
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public suspend fun isValid(
        formValue: T,
        path: Path,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Boolean = validate(formValue, path, externalContexts, runCriteria).containsNoErrors()

    /**
     * Returns whether the parts of the form value [formValue] matching [path] are valid according
     * to their schemas. These parts are said to be valid if they contain no validation errors.
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws InvalidPathException If [path] matches no schemas.
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public suspend fun isValid(
        formValue: T,
        path: String,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Boolean = isValid(formValue, AbsolutePath(path), externalContexts, runCriteria)

    /**
     * Returns whether all parts of the form value [formValue] are valid according to their schemas.
     * These parts are said to be valid if they contain no validation errors.
     *
     * A map of [external contexts][externalContexts] may be provided for validations that depend on
     * them.
     *
     * Provide a [run criteria][runCriteria] to specify whether to run schema-internal and/or
     * external validations. By default, external validations only run if no errors were found when
     * running schema-internal validations.
     *
     * @throws ValidationFailureException If an exception occurs while running a validation.
     */
    public suspend fun isValid(
        formValue: T,
        externalContexts: ExternalContexts? = null,
        runCriteria: ValidateRunCriteria = ValidateRunCriteria.ExternalIfInternalValid,
    ): Boolean = isValid(formValue, AbsolutePath.MATCH_ALL, externalContexts, runCriteria)
}
