@file:JvmName("FormUtils")

package tech.ostack.kform

import io.github.oshai.kotlinlogging.KotlinLogging
import kotlin.jvm.JvmName
import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.flow.*
import tech.ostack.kform.internal.*
import tech.ostack.kform.internal.INITIAL_VALUE
import tech.ostack.kform.internal.SchemaEventsNoOpBus
import tech.ostack.kform.internal.computationDependenciesInfo
import tech.ostack.kform.internal.schemaInfoImpl
import tech.ostack.kform.internal.validateComputation
import tech.ostack.kform.internal.valueInfoImpl

/** Logger used by the form util. */
private val logger = KotlinLogging.logger {}

/** External contexts. */
public typealias ExternalContexts = Map<String, Any?>

/** External validations. */
public typealias ExternalValidations = Map<PathOrString, Iterable<Validation<*>>>

/**
 * Validates that the provided [path] points to a schema of [formSchema].
 *
 * @throws InvalidPathException If the path is invalid.
 */
public fun validatePath(formSchema: Schema<*>, path: Path): Unit =
    path.toAbsolutePath().let { absolutePath ->
        if (!schemaInfoImpl(formSchema, absolutePath).any()) {
            throw InvalidPathException(absolutePath, "No schema matches this path.")
        }
    }

/**
 * Validates all validations of the provided [formSchema] by checking that all validation
 * dependencies are valid (i.e. that they point to valid locations and have valid types).
 *
 * @throws InvalidDependencyPathException If a validation has an invalid dependency path.
 * @throws InvalidDependencyTypeException If a validation has an invalid dependency type.
 */
public fun validateSchemaValidations(formSchema: Schema<*>) {
    for ((schema, path) in schemaInfo(formSchema, AbsolutePath.MATCH_ALL)) {
        for (validation in schema.validations) {
            validateComputation(formSchema, path, validation)
        }
    }
}

/**
 * Validates the provided [external validations][externalValidations] in the context of the given
 * [formSchema] by checking that all validation dependencies are valid (i.e. that they point to
 * valid locations and have valid types).
 *
 * @throws InvalidPathException If an external validation path is invalid.
 * @throws InvalidDependencyPathException If a validation has an invalid dependency path.
 * @throws InvalidDependencyTypeException If a validation has an invalid dependency type.
 */
public fun validateExternalValidations(
    formSchema: Schema<*>,
    externalValidations: Map<PathOrString, Iterable<Validation<*>>>,
) {
    for ((path, validations) in externalValidations) {
        val absolutePath = path.toAbsolutePath()
        validatePath(formSchema, absolutePath)
        for (validation in validations) {
            validateComputation(formSchema, absolutePath, validation)
        }
    }
}

/**
 * Returns whether there exists at least one schema within [formSchema] matching [path].
 *
 * Paths that match no schemas are deemed invalid, and most functions called with them will throw.
 */
public fun isValidPath(formSchema: Schema<*>, path: Path): Boolean =
    schemaInfoImpl(formSchema, AbsolutePath(path)).any()

/**
 * Returns whether there exists at least one schema within [formSchema] matching [path].
 *
 * Paths that match no schemas are deemed invalid, and most functions called with them will throw.
 */
public fun isValidPath(formSchema: Schema<*>, path: String): Boolean =
    isValidPath(formSchema, AbsolutePath(path))

/**
 * Returns a sequence of information about the schemas within [formSchema] matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public fun schemaInfo(
    formSchema: Schema<*>,
    path: Path = AbsolutePath.MATCH_ALL,
): Sequence<SchemaInfo<*>> =
    path.toAbsolutePath().let { absolutePath ->
        validatePath(formSchema, absolutePath)
        schemaInfoImpl(formSchema, absolutePath)
    }

/**
 * Returns a sequence of information about the schemas within [formSchema] matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public fun schemaInfo(formSchema: Schema<*>, path: String): Sequence<SchemaInfo<*>> =
    schemaInfo(formSchema, AbsolutePath(path))

/**
 * Returns the single schema within [formSchema] matching [path].
 *
 * To get information about all schemas matching a path use [schemaInfo] instead.
 *
 * @throws InvalidPathException If [path] matches no schemas or more than one schema.
 */
public fun schema(formSchema: Schema<*>, path: Path): Schema<*> =
    path.toAbsolutePath().let { absolutePath ->
        try {
            schemaInfo(formSchema, absolutePath).single().schema
        } catch (_: IllegalArgumentException) {
            throw InvalidPathException(absolutePath, "Path matches more than one schema.")
        }
    }

/**
 * Returns the single schema within [formSchema] matching [path].
 *
 * To get information about all schemas matching a path use [schemaInfo] instead.
 *
 * @throws InvalidPathException If [path] matches no schemas or more than one schema.
 */
public fun schema(formSchema: Schema<*>, path: String): Schema<*> =
    schema(formSchema, AbsolutePath(path))

/**
 * Returns a flow of information about the parts of the form value [formValue] (with schema
 * [formSchema]) matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public fun <T> valueInfo(
    formSchema: Schema<T>,
    formValue: T,
    path: Path = AbsolutePath.MATCH_ALL,
): Flow<ValueInfo<*>> =
    path.toAbsolutePath().let { absolutePath ->
        validatePath(formSchema, absolutePath)
        valueInfoImpl(formSchema, formValue, absolutePath)
    }

/**
 * Returns a flow of information about the parts of the form value [formValue] (with schema
 * [formSchema]) matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public fun <T> valueInfo(formSchema: Schema<T>, formValue: T, path: String): Flow<ValueInfo<*>> =
    valueInfo(formSchema, formValue, AbsolutePath(path))

/**
 * Returns whether there exists a part of the form value [formValue] (with schema [formSchema])
 * matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public suspend fun <T> has(formSchema: Schema<T>, formValue: T, path: Path): Boolean =
    valueInfo(formSchema, formValue, path).firstOrNull() != null

/**
 * Returns whether there exists a part of the form value [formValue] (with schema [formSchema])
 * matching [path].
 *
 * @throws InvalidPathException If [path] matches no schemas.
 */
public suspend fun <T> has(formSchema: Schema<T>, formValue: T, path: String): Boolean =
    has(formSchema, formValue, AbsolutePath(path))

/**
 * Returns the single part of the form value [formValue] (with schema [formSchema]) matching [path].
 *
 * To get information about multiple parts of a form value at once, use [valueInfo] instead.
 *
 * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
 * @throws IllegalStateException If [path] matches more than one value.
 * @throws NoSuchElementException If no part of [formValue] matches [path].
 */
public suspend fun <T> get(formSchema: Schema<T>, formValue: T, path: Path): Any? =
    path.toAbsolutePath().let { absolutePath ->
        if (absolutePath.hasAnyWildcard()) {
            throw InvalidPathException(absolutePath, "Path cannot contain wildcards.")
        }
        validatePath(formSchema, absolutePath)
        valueInfoImpl(formSchema, formValue, absolutePath).single().value
    }

/**
 * Returns the single part of the form value [formValue] (with schema [formSchema]) matching [path].
 *
 * To get information about multiple parts of a form value at once, use [valueInfo] instead.
 *
 * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
 * @throws IllegalStateException If [path] matches more than one value.
 * @throws NoSuchElementException If no part of [formValue] matches [path].
 */
public suspend fun <T> get(formSchema: Schema<T>, formValue: T, path: String): Any? =
    get(formSchema, formValue, AbsolutePath(path))

/**
 * Returns a clone (deep copy) of the single part of the form value [formValue] (with schema
 * [formSchema]) matching [path].
 *
 * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
 * @throws IllegalStateException If [path] matches more than one value.
 * @throws NoSuchElementException If no part of [formValue] matches [path].
 */
public suspend fun <T> getClone(formSchema: Schema<T>, formValue: T, path: Path): Any? =
    path.toAbsolutePath().let { absolutePath ->
        if (absolutePath.hasAnyWildcard()) {
            throw InvalidPathException(absolutePath, "Path cannot contain wildcards.")
        }
        validatePath(formSchema, absolutePath)
        @Suppress("UNCHECKED_CAST")
        val info = valueInfoImpl(formSchema, formValue, absolutePath).single() as ValueInfo<Any?>
        info.schema.clone(info.value)
    }

/**
 * Returns a clone (deep copy) of the single part of the form value [formValue] (with schema
 * [formSchema]) matching [path].
 *
 * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
 * @throws IllegalStateException If [path] matches more than one value.
 * @throws NoSuchElementException If no part of [formValue] matches [path].
 */
public suspend fun <T> getClone(formSchema: Schema<T>, formValue: T, path: String): Any? =
    getClone(formSchema, formValue, AbsolutePath(path))

/**
 * Sets values at [path] that are part of the form value [formValue] (with schema [formSchema]) with
 * value [toSet].
 *
 * If the path has a trailing non-recursive wildcard, then all existing children of its parent value
 * are set to [toSet]. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`; setting the value
 * `5` at `"/list/∗"` will cause `"/list"` to end up with `[5, 5, 5]`.
 *
 * Setting a value on a path with a trailing recursive wildcard is considered equivalent to setting
 * the value on said path without such wildcard. E.g. setting the value at `"/x/∗∗"` is equivalent
 * to setting the same value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schemas, or when attempting to set the root
 *   value.
 */
public suspend fun <T> set(formSchema: Schema<T>, formValue: T, path: Path, toSet: Any?): Unit =
    path.toAbsolutePath().let { absolutePath ->
        // Normalise the path by removing trailing recursive wildcards
        val normalizedPath =
            if (absolutePath.lastFragment is AbsolutePathFragment.RecursiveWildcard)
                absolutePath.parent()
            else absolutePath

        if (normalizedPath.isRoot) {
            throw InvalidPathException(normalizedPath, "Cannot set root value.")
        }
        validatePath(formSchema, normalizedPath)

        val fragment = normalizedPath.lastFragment!!
        val parentPath = normalizedPath.parent()
        valueInfoImpl(formSchema, formValue, parentPath).collect { parentInfo ->
            setOnParent(parentInfo, fragment, toSet, SchemaEventsNoOpBus)
        }
    }

/**
 * Sets values at [path] that are part of the form value [formValue] (with schema [formSchema]) with
 * value [toSet].
 *
 * If the path has a trailing non-recursive wildcard, then all existing children of its parent value
 * are set to [toSet]. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`; setting the value
 * `5` at `"/list/∗"` will cause `"/list"` to end up with `[5, 5, 5]`.
 *
 * Setting a value on a path with a trailing recursive wildcard is considered equivalent to setting
 * the value on said path without such wildcard. E.g. setting the value at `"/x/∗∗"` is equivalent
 * to setting the same value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schemas, or when attempting to set the root
 *   value.
 */
public suspend fun <T> set(formSchema: Schema<T>, formValue: T, path: String, toSet: Any?): Unit =
    set(formSchema, formValue, AbsolutePath(path), toSet)

/**
 * Resets the values at [path] that are part of the form value [formValue] (with schema
 * [formSchema]) to their initial value.
 *
 * If the path has a trailing non-recursive wildcard, then all existing children of its parent value
 * will have their value reset. E.g. assume that the list `[1, 2, 3]` exists at `"/list"` and that
 * the schema of `"/list/∗"` has an initial value of `0`; resetting `"/list/∗"` will thus cause
 * `"/list"` to end up with `[0, 0, 0]`.
 *
 * Resetting the value on a path with a trailing recursive wildcard is considered equivalent to
 * resetting the value on said path without such wildcard. E.g. resetting the value at `"/x/∗∗"` is
 * equivalent to resetting the value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schemas, or when attempting to reset the root
 *   value.
 */
public suspend fun <T> reset(formSchema: Schema<T>, formValue: T, path: Path): Unit =
    set(formSchema, formValue, AbsolutePath(path), INITIAL_VALUE)

/**
 * Resets the values at [path] that are part of the form value [formValue] (with schema
 * [formSchema]) to their initial value.
 *
 * If the path has a trailing non-recursive wildcard, then all existing children of its parent value
 * will have their value reset. E.g. assume that the list `[1, 2, 3]` exists at `"/list"` and that
 * the schema of `"/list/∗"` has an initial value of `0`; resetting `"/list/∗"` will thus cause
 * `"/list"` to end up with `[0, 0, 0]`.
 *
 * Resetting the value on a path with a trailing recursive wildcard is considered equivalent to
 * resetting the value on said path without such wildcard. E.g. resetting the value at `"/x/∗∗"` is
 * equivalent to resetting the value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schemas, or when attempting to reset the root
 *   value.
 */
public suspend fun <T> reset(formSchema: Schema<T>, formValue: T, path: String): Unit =
    reset(formSchema, formValue, AbsolutePath(path))

/**
 * Removes the values matching [path] that are part of the form value [formValue] (with schema
 * [formSchema]) from their parent collection(s).
 *
 * It is possible to clear a collection by providing a path with a trailing wildcard.
 *
 * Removing the value on a path with a trailing recursive wildcard is considered equivalent to
 * removing the value on said path without such wildcard. E.g. removing the value at `"/x/∗∗"` is
 * equivalent to removing the value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schema paths, when attempting to remove the
 *   root value, or when a parent of [path] is not a collection.
 */
public suspend fun <T> remove(formSchema: Schema<T>, formValue: T, path: Path): Unit =
    path.toAbsolutePath().let { absolutePath ->
        // Normalise the path by removing trailing recursive wildcards
        val normalizedPath =
            if (absolutePath.lastFragment is AbsolutePathFragment.RecursiveWildcard)
                absolutePath.parent()
            else absolutePath

        if (normalizedPath.isRoot) {
            throw InvalidPathException(normalizedPath, "Cannot remove root value.")
        }
        validatePath(formSchema, normalizedPath)

        val parentPath = normalizedPath.parent()
        val parentSchemasInfo = schemaInfoImpl(formSchema, parentPath)
        for ((parentSchema, _, parentQueriedPath) in parentSchemasInfo) {
            if (parentSchema !is CollectionSchema<*, *>) {
                throw InvalidPathException(
                    normalizedPath,
                    "Schema at '$parentQueriedPath' is not a collection schema.",
                )
            }
        }

        val fragment = normalizedPath.lastFragment!!
        for (parentSchemaInfo in parentSchemasInfo) {
            valueInfoImpl(formSchema, formValue, parentSchemaInfo.queriedPath).collect { parentInfo
                ->
                removeFromParent(parentInfo, fragment, SchemaEventsNoOpBus)
            }
        }
    }

/**
 * Removes the values matching [path] that are part of the form value [formValue] (with schema
 * [formSchema]) from their parent collection(s).
 *
 * It is possible to clear a collection by providing a path with a trailing wildcard.
 *
 * Removing the value on a path with a trailing recursive wildcard is considered equivalent to
 * removing the value on said path without such wildcard. E.g. removing the value at `"/x/∗∗"` is
 * equivalent to removing the value at `"/x"`.
 *
 * @throws InvalidPathException If [path] matches no schema paths, when attempting to remove the
 *   root value, or when a parent of [path] is not a collection.
 */
public suspend fun <T> remove(formSchema: Schema<T>, formValue: T, path: String): Unit =
    remove(formSchema, formValue, AbsolutePath(path))

/**
 * Validates the parts of the form value [formValue] matching [path] against [formSchema]. Returns a
 * flow of found [validation issues][LocatedValidationIssue].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validate(
    formSchema: Schema<T>,
    formValue: T,
    path: Path,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    path.toAbsolutePath().let { absolutePath ->
        validatePath(formSchema, absolutePath)
        flow {
            valueInfoImpl(formSchema, formValue, absolutePath).collect { info ->
                for (validation in info.schema.validations) {
                    emitAll(
                        @Suppress("UNCHECKED_CAST")
                        runValidation(
                            formSchema,
                            formValue,
                            externalContexts,
                            validation as Validation<Any?>,
                            info,
                        )
                    )
                }
            }
        }
    }

/**
 * Validates the parts of the form value [formValue] matching [path] against [formSchema]. Returns a
 * flow of found [validation issues][LocatedValidationIssue].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validate(
    formSchema: Schema<T>,
    formValue: T,
    path: String,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    validate(formSchema, formValue, AbsolutePath(path), externalContexts)

/**
 * Validates all parts of the form value [formValue] against [formSchema]. Returns a flow of found
 * [validation issues][LocatedValidationIssue].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validate(
    formSchema: Schema<T>,
    formValue: T,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    validate(formSchema, formValue, AbsolutePath.MATCH_ALL, externalContexts)

/**
 * Validates the parts of the form value [formValue] matching [path] with schema [formSchema]
 * against a map of external validations [externalValidations].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validateExternally(
    formSchema: Schema<T>,
    formValue: T,
    path: Path,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    path.toAbsolutePath().let { absolutePath ->
        validatePath(formSchema, absolutePath)
        flow {
            for ((validatingPath, validations) in externalValidations) {
                val validatingAbsolutePath = validatingPath.toAbsolutePath()
                if (validatingAbsolutePath in absolutePath) {
                    valueInfoImpl(formSchema, formValue, validatingAbsolutePath).collect { info ->
                        for (validation in validations) {
                            emitAll(
                                @Suppress("UNCHECKED_CAST")
                                runValidation(
                                    formSchema,
                                    formValue,
                                    externalContexts,
                                    validation as Validation<Any?>,
                                    info,
                                )
                            )
                        }
                    }
                }
            }
        }
    }

/**
 * Validates the parts of the form value [formValue] matching [path] with schema [formSchema]
 * against a map of external validations [externalValidations].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validateExternally(
    formSchema: Schema<T>,
    formValue: T,
    path: String,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    validateExternally(
        formSchema,
        formValue,
        AbsolutePath(path),
        externalValidations,
        externalContexts,
    )

/**
 * Validates all parts of the form value [formValue] with schema [formSchema] against a map of
 * external validations [externalValidations].
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public fun <T> validateExternally(
    formSchema: Schema<T>,
    formValue: T,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Flow<LocatedValidationIssue> =
    validateExternally(
        formSchema,
        formValue,
        AbsolutePath.MATCH_ALL,
        externalValidations,
        externalContexts,
    )

/** Runs a single validation and returns a flow over its validation issues. */
private fun <T, TValidated> runValidation(
    formSchema: Schema<T>,
    formValue: T,
    externalContexts: ExternalContexts?,
    validation: Validation<TValidated>,
    info: ValueInfo<TValidated>,
): Flow<LocatedValidationIssue> = flow {
    val validationContext =
        ValidationContext(
            info.value,
            info.schema,
            info.path,
            info.schemaPath,
            computationDependenciesInfo(formSchema, formValue, info.path, validation.dependencies),
            validation.externalContextDependencies.associateWith { externalContexts?.get(it) },
        )

    fun handleException(ex: Throwable) {
        if (ex !is CancellationException) {
            throw ValidationFailureException(info.path, validation, ex)
        }
    }

    // Wrap code in `try/catch` since we're calling user code (`validate`) and an error may
    // occur when **creating** the flow. Typically, however, if an error occurs, it will be
    // within the flow.
    var issuesFlow: Flow<ValidationIssue>? = null
    try {
        issuesFlow = validation.run { validationContext.validate() }
    } catch (ex: Throwable) {
        handleException(ex)
    }
    // If the validation throws, we still emit the issues up to the point it threw
    issuesFlow
        ?.catch { ex -> handleException(ex) }
        ?.collect { issue -> emit(LocatedValidationIssue(info.path, validation, issue)) }
}

/**
 * Returns whether the parts of the form value [formValue] (with schema [formSchema]) matching
 * [path] are valid according to their schemas. These parts are said to be valid if they contain no
 * validation errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValid(
    formSchema: Schema<T>,
    formValue: T,
    path: Path,
    externalContexts: ExternalContexts? = null,
): Boolean = validate(formSchema, formValue, path, externalContexts).containsNoErrors()

/**
 * Returns whether the parts of the form value [formValue] (with schema [formSchema]) matching
 * [path] are valid according to their schemas. These parts are said to be valid if they contain no
 * validation errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValid(
    formSchema: Schema<T>,
    formValue: T,
    path: String,
    externalContexts: ExternalContexts? = null,
): Boolean = isValid(formSchema, formValue, AbsolutePath(path), externalContexts)

/**
 * Returns whether all parts of the form value [formValue] (with schema [formSchema]) are valid
 * according to their schemas. These parts are said to be valid if they contain no validation
 * errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValid(
    formSchema: Schema<T>,
    formValue: T,
    externalContexts: ExternalContexts? = null,
): Boolean = isValid(formSchema, formValue, AbsolutePath.MATCH_ALL, externalContexts)

/**
 * Returns whether the parts of the form value [formValue] (with schema [formSchema]) matching
 * [path] are valid according to the provided [external validations][externalValidations]. These
 * parts are said to be valid if they contain no validation errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValidExternally(
    formSchema: Schema<T>,
    formValue: T,
    path: Path,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Boolean =
    validateExternally(formSchema, formValue, path, externalValidations, externalContexts)
        .containsNoErrors()

/**
 * Returns whether the parts of the form value [formValue] (with schema [formSchema]) matching
 * [path] are valid according to the provided [external validations][externalValidations]. These
 * parts are said to be valid if they contain no validation errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws InvalidPathException If [path] matches no schemas.
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValidExternally(
    formSchema: Schema<T>,
    formValue: T,
    path: String,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Boolean =
    isValidExternally(
        formSchema,
        formValue,
        AbsolutePath(path),
        externalValidations,
        externalContexts,
    )

/**
 * Returns whether all parts of the form value [formValue] (with schema [formSchema]) are valid
 * according to the provided [external validations][externalValidations]. These parts are said to be
 * valid if they contain no validation errors.
 *
 * A map of [external contexts][externalContexts] may be provided for validations that depend on
 * them.
 *
 * @throws ValidationFailureException If an exception occurs while running a validation.
 */
public suspend fun <T> isValidExternally(
    formSchema: Schema<T>,
    formValue: T,
    externalValidations: ExternalValidations,
    externalContexts: ExternalContexts? = null,
): Boolean =
    isValidExternally(
        formSchema,
        formValue,
        AbsolutePath.MATCH_ALL,
        externalValidations,
        externalContexts,
    )
