package tech.ostack.kform

import kotlin.jvm.JvmOverloads
import kotlin.reflect.KType
import tech.ostack.kform.internal.kTypeToString

/**
 * Exception occurring at the provided [path] with the given [message] and optional [cause].
 *
 * @property path Path where the exception occurred.
 */
public open class AtPathException
@JvmOverloads
constructor(public val path: AbsolutePath, message: String, cause: Throwable? = null) :
    RuntimeException("At '$path': $message", cause)

/** Exception with message [message] thrown when an invalid [path] was provided. */
public open class InvalidPathException(path: AbsolutePath, message: String) :
    AtPathException(path, message)

/**
 * Exception thrown when a validation fails to run via the [form validator][FormValidator] or form
 * utilities due to an exception being thrown while running it.
 *
 * The original thrown exception is accessible as the [cause] of this exception.
 *
 * @property validation Validation in which the error occurred.
 */
public class ValidationFailureException(
    path: AbsolutePath,
    public val validation: Validation<*>,
    cause: Throwable,
) : AtPathException(path, "Failed to run validation '$validation'", cause)

/**
 * Exception thrown when a [computation] at path [path] has a dependency with an invalid path.
 *
 * @property computation Computation containing the invalid dependency.
 * @property dependencyKey Key of the invalid dependency.
 * @property dependencyPath Path of the invalid dependency.
 */
public class InvalidDependencyPathException(
    path: AbsolutePath,
    public val computation: Computation,
    public val dependencyKey: String,
    public val dependencyPath: Path,
) :
    InvalidPathException(
        path,
        "At '$computation': At dependency '$dependencyKey': Invalid path '$dependencyPath'.",
    )

/**
 * Exception thrown when a [computation] at path [path] has a dependency with an invalid type.
 *
 * @property computation Computation containing the invalid dependency.
 * @property expectedType Expected type of the dependency.
 * @property dependencyKey Key of the invalid dependency.
 * @property dependencyType Type of the invalid dependency.
 */
public class InvalidDependencyTypeException(
    path: AbsolutePath,
    public val computation: Computation,
    public val expectedType: String,
    public val dependencyKey: String,
    public val dependencyType: KType,
) :
    InvalidPathException(
        path,
        "At '$computation': At dependency '$dependencyKey': " +
            "Expected type: '$expectedType', actual type: '${kTypeToString(dependencyType)}'.",
    )

/**
 * Exception thrown when a [stateful computation][computation] at path [path] has an invalid path to
 * observe [toObserve].
 *
 * @property computation Computation containing the invalid path to observe.
 * @property toObserve Path to observe.
 */
public class InvalidPathToObserveException(
    path: AbsolutePath,
    public val computation: StatefulComputation<*, *>,
    public val toObserve: Path,
) : InvalidPathException(path, "At '$computation': Invalid path to observe '$toObserve'.")

/**
 * Exception thrown when attempting to access a dependency with key [dependencyKey] that could not
 * be found during the execution of a computation.
 *
 * @property dependencyKey Key of the dependency as defined in the computation.
 */
public class DependencyNotFoundException(public val dependencyKey: String) :
    RuntimeException("Dependency '$dependencyKey' was not found.")

/**
 * Exception thrown when attempting to access an external context with name [externalContextName]
 * that could not be found during the execution of a computation.
 *
 * @property externalContextName Name of the external context as defined in the computation.
 */
public class ExternalContextNotFoundException(public val externalContextName: String) :
    RuntimeException("External context '$externalContextName' was not found.")
