package tech.ostack.kform

import kotlin.js.JsName
import kotlin.jvm.JvmName
import kotlin.properties.PropertyDelegateProvider
import kotlin.properties.ReadOnlyProperty
import kotlin.reflect.typeOf

/**
 * Context provided to a computed value when running it, contains the values of the computed value's
 * dependencies at the time the computed value was executed.
 */
public open class ComputedValueContext(
    schema: Schema<*>,
    /** Path of the value being computed. */
    path: AbsolutePath,
    /** Path of the schema of the value being computed. */
    schemaPath: AbsolutePath,
    /**
     * Value information for each dependency of the computed value. Mapping of keys as declared in
     * the [computed value dependencies][ComputedValue.dependencies] to their respective value
     * information.
     */
    dependenciesInfo: DependenciesValueInfo,
    externalContexts: ExternalContexts,
) : ComputationContext(schema, path, schemaPath, dependenciesInfo, externalContexts)

/**
 * Computed value for values of type [T].
 *
 * Computed values should depend on other values. These dependencies may be defined via the
 * [dependency] function, by providing the path of the dependency relative to the value being
 * computed.
 *
 * Example computed value which computes the sum between two other values:
 * ```kotlin
 * object MySum : ComputedValue<Int>() {
 *     private val ComputedValueContext.value1: Int? by dependency("../value1")
 *     private val ComputedValueContext.value2: Int? by dependency("../value2")
 *
 *     override suspend fun ComputedValueContext.compute(): Int = (value1 ?: 0) + (value2 ?: 0)
 * }
 * ```
 */
@JsName("ComputedValueKt")
public abstract class ComputedValue<out T> : Computation {
    /** Schema of the value being computed. */
    protected inline val ComputedValueContext.schema: Schema<@UnsafeVariance T>
        get() = schema()

    /**
     * Dependencies of the computed value. Mapping of keys to the paths this computed value depends
     * on. Keys can be used within a [ComputedValueContext] to access the value of the dependencies.
     *
     * Computed value dependency paths may contain a single recursive wildcard at the end,
     * indicating that the computed value should be reevaluated by the [form manager][FormManager]
     * whenever a child value of the dependency changes. Otherwise, a dependency must contain no
     * wildcards.
     */
    public override val dependencies: Map<String, DependencyInfo> = hashMapOf()

    /**
     * Set of external context dependencies of the computed value.
     *
     * Each entry of this set represents the name of the external context being depended upon.
     */
    public override val externalContextDependencies: Set<String> = hashSetOf()

    /**
     * Declares a dependency to [path], accessible in the computed value's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    @JvmName("reifiedAddDependency")
    protected inline fun <reified TDependency> addDependency(dependencyKey: String, path: Path) {
        (dependencies as MutableMap<String, DependencyInfo>)[dependencyKey] =
            DependencyInfo(path, typeOf<TDependency>())
    }

    /**
     * Declares a dependency to [path], accessible in the computed value's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addDependency(dependencyKey: String, path: Path): Unit =
        addDependency<Any?>(dependencyKey, path)

    /**
     * Declares a dependency to [path], accessible in the computed value's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    @JvmName("reifiedAddDependency")
    protected inline fun <reified TDependency> addDependency(
        dependencyKey: String,
        path: String,
    ): Unit = addDependency<TDependency>(dependencyKey, Path(path))

    /**
     * Declares a dependency to [path], accessible in the computed value's context via key
     * [dependencyKey].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addDependency(dependencyKey: String, path: String): Unit =
        addDependency<Any?>(dependencyKey, path)

    /**
     * Declares an external context dependency to [externalContextName].
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun addExternalContextDependency(externalContextName: String) {
        (externalContextDependencies as MutableSet<String>).add(externalContextName)
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ComputedValueContext].
     *
     * If the dependency could not be found in the form during the execution of the computed value,
     * then accessing this dependency will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.dependencyKey: Type? by dependencyOrNull(path)
     * ```
     *
     * Computed value dependency paths may contain a single recursive wildcard at the end,
     * indicating that the computed value should be reevaluated by the [form manager][FormManager]
     * whenever a child value of the dependency changes. Otherwise, a dependency must contain no
     * wildcards.
     */
    protected inline fun <reified TDependency> dependencyOrNull(
        path: Path,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TDependency?>,
    > = PropertyDelegateProvider { computedValue, property ->
        computedValue.addDependency<TDependency>(dependencyKey ?: property.name, path)
        // Return a delegate to the dependency value within a [ComputedValueContext]
        ReadOnlyProperty { ctx, prop -> ctx.dependencyOrNull(dependencyKey ?: prop.name) }
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ComputedValueContext].
     *
     * If the dependency could not be found in the form during the execution of the computed value,
     * then accessing this dependency will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.dependencyKey: Type? by dependencyOrNull(path)
     * ```
     *
     * Computed value dependency paths may contain a single recursive wildcard at the end,
     * indicating that the computed value should be reevaluated by the [form manager][FormManager]
     * whenever a child value of the dependency changes. Otherwise, a dependency must contain no
     * wildcards.
     */
    protected inline fun <reified TDependency> dependencyOrNull(
        path: String,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TDependency?>,
    > = dependencyOrNull(Path(path), dependencyKey)

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ComputedValueContext].
     *
     * If the dependency could not be found in the form during the execution of the computed value,
     * then accessing this dependency will throw [DependencyNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.dependencyKey: Type by dependency(path)
     * ```
     *
     * Computed value dependency paths may contain a single recursive wildcard at the end,
     * indicating that the computed value should be reevaluated by the [form manager][FormManager]
     * whenever a child value of the dependency changes. Otherwise, a dependency must contain no
     * wildcards.
     */
    protected inline fun <reified TDependency> dependency(
        path: Path,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TDependency>,
    > = PropertyDelegateProvider { computedValue, property ->
        computedValue.addDependency<TDependency>(dependencyKey ?: property.name, path)
        // Return a delegate to the dependency value within a [ComputedValueContext]
        ReadOnlyProperty { ctx, prop -> ctx.dependency(dependencyKey ?: prop.name) }
    }

    /**
     * Function used to declare a dependency to a [path] and delegate access to its value within a
     * [ComputedValueContext].
     *
     * If the dependency could not be found in the form during the execution of the computed value,
     * then accessing this dependency will throw [DependencyNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.dependencyKey: Type by dependency(path)
     * ```
     *
     * Computed value dependency paths may contain a single recursive wildcard at the end,
     * indicating that the computed value should be reevaluated by the [form manager][FormManager]
     * whenever a child value of the dependency changes. Otherwise, a dependency must contain no
     * wildcards.
     */
    protected inline fun <reified TDependency> dependency(
        path: String,
        dependencyKey: String? = null,
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TDependency>,
    > = dependency(Path(path), dependencyKey)

    /**
     * Function used to declare a dependency to an external context and delegate access to its value
     * within a [ComputedValueContext].
     *
     * If the external context could not be found during the execution of the computed value, then
     * accessing this external context will return `null`.
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.externalContextName: Type? by externalContextOrNull()
     * ```
     */
    protected fun <TContext> externalContextOrNull(
        externalContextName: String? = null
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TContext?>,
    > = PropertyDelegateProvider { computedValue, property ->
        computedValue.addExternalContextDependency(externalContextName ?: property.name)
        // Return a delegate to the external context dependency value within a
        // [ComputedValueContext]
        ReadOnlyProperty { ctx, prop ->
            ctx.externalContextOrNull(externalContextName ?: prop.name)
        }
    }

    /**
     * Function used to declare a dependency to an external context and delegate access to its value
     * within a [ComputedValueContext].
     *
     * If the external context could not be found during the execution of the computed value, then
     * accessing this external context will throw [ExternalContextNotFoundException].
     *
     * It should be used as follows:
     * ```kotlin
     * private val ComputedValueContext.externalContextName: Type by externalContext()
     * ```
     */
    protected fun <TContext> externalContext(
        externalContextName: String? = null
    ): PropertyDelegateProvider<
        ComputedValue<@UnsafeVariance T>,
        ReadOnlyProperty<ComputedValueContext, TContext>,
    > = PropertyDelegateProvider { computedValue, property ->
        computedValue.addExternalContextDependency(externalContextName ?: property.name)
        // Return a delegate to the external context dependency value within a
        // [ComputedValueContext]
        ReadOnlyProperty { ctx, prop -> ctx.externalContext(externalContextName ?: prop.name) }
    }

    /**
     * Runs the computation within a [ComputedValueContext] containing the value of all declared
     * dependencies. Returns the computed value.
     */
    public abstract suspend fun ComputedValueContext.compute(): T

    // Default [toString] implementation
    public override fun toString(): String = this::class.simpleName ?: super.toString()
}

/**
 * Computed value for values of type [T] where the [form manager][FormManager] maintains a computed
 * value state of type [TState] for each managed value.
 *
 * Stateful computed values are useful when computed values require an expensive computation over
 * data and if it is possible to save the result of such expensive computation and tweak it as data
 * changes instead of running the expensive computation all over again.
 *
 * As an example, imagine that we have a list of people and that we want to compute the average age
 * of all people. Instead of iterating over the whole list every time a person is added or removed,
 * we can use a stateful computed value to save, as state, the sum of all ages, and simply tweak
 * this sum as people are added or removed. Having access to the sum of all ages as state allows us
 * to implement the computed value function with a complexity of O(1) as opposed to O(N).
 *
 * The following snippet implements the example above:
 * ```kotlin
 * object AvgAgeNotOverMax : StatefulComputedValue<Int, Int>() {
 *     private val ComputedValueContext.people: List<Person> by dependency("../people")
 *
 *     override suspend fun ComputedValueContext.initState(): Int =
 *         people.fold(0) { sum, person -> sum + person.age }
 *
 *     private val ageObserver by observe<Int>("../people/∗/age") { agesSum, event ->
 *         when (event) {
 *             is ValueEvent.Init<Int> -> agesSum + event.newValue
 *             is ValueEvent.Change<Int> -> agesSum + event.newValue - event.oldValue
 *             is ValueEvent.Destroy<Int> -> agesSum - event.oldValue
 *             else -> agesSum
 *         }
 *     }
 *
 *     override suspend fun ComputedValueContext.computeFromState(state: Int) = state / people.size
 * }
 * ```
 */
@JsName("StatefulComputedValueKt")
public abstract class StatefulComputedValue<out T, TState> :
    StatefulComputation<ComputedValueContext, TState>, ComputedValue<T>() {
    /**
     * List of observers.
     *
     * Each observer contains a path to be observed and a function to update the state of the
     * computed value, which will be called whenever an event with a path matching the one being
     * observed occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     */
    public override val observers: List<Observer<Any?, TState>> = mutableListOf()

    /**
     * Adds an [observer] to the path [observer.toObserve][Observer.toObserve] to update the
     * computed value state via [observer.updateState][Observer.updateState] whenever an event with
     * a path matching [observer.toObserve][Observer.toObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(observer: Observer<TValue, TState>) {
        @Suppress("UNCHECKED_CAST")
        (observers as MutableList<Observer<*, TState>>) += observer
    }

    /**
     * Adds an observer to the path [pathToObserve] to update the computed value state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(
        pathToObserve: Path,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): Unit = addObserver(Observer(pathToObserve, updateState))

    /**
     * Adds an observer to the path [pathToObserve] to update the computed value state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * **NOTE**: This method must only be called during the initialization of the class.
     */
    protected fun <TValue> addObserver(
        pathToObserve: String,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): Unit = addObserver(Path(pathToObserve), updateState)

    /**
     * Function used to observe the path [pathToObserve] and update the computed value state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * It should be used as follows:
     * ```kotlin
     * private val observer by observe<Type>(path) { state, event ->
     *     when (event) {
     *         is ValueEvent.Init<Type>      -> // Compute new state
     *         is ValueEvent.Change<Type>    -> // Compute new state
     *         is ValueEvent.Destroy<Type>   -> // Compute new state
     *
     *         // These events only occur when observing a collection:
     *         is ValueEvent.Add<Type, *>    -> // Compute new state
     *         is ValueEvent.Remove<Type, *> -> // Compute new state
     *     }
     * }
     * ```
     *
     * The provided function must return the new computed value state which, unless it hasn't
     * changed, should be different ([equals]-wise) from the previous computed value state (as such,
     * when using an object as state, a new instance should be returned when the state is updated).
     * If the new state differs from the old one (using [equals]), the value will need to be
     * recomputed by the [form manager][FormManager] using [computeFromState].
     */
    protected fun <TValue> observe(
        pathToObserve: Path,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): PropertyDelegateProvider<
        StatefulComputedValue<@UnsafeVariance T, TState>,
        ReadOnlyProperty<StatefulComputedValue<@UnsafeVariance T, TState>, Observer<TValue, TState>>,
    > = PropertyDelegateProvider { computedValue, _ ->
        val observer = Observer(pathToObserve, updateState)
        computedValue.addObserver(observer)
        ReadOnlyProperty { _, _ -> observer }
    }

    /**
     * Function used to observe the path [pathToObserve] and update the computed value state via
     * [updateState] whenever an event with a path matching [pathToObserve] occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     *
     * It should be used as follows:
     * ```kotlin
     * private val observer by observe<Type>(path) { state, event ->
     *     when (event) {
     *         is ValueEvent.Init<Type>      -> // Compute new state
     *         is ValueEvent.Change<Type>    -> // Compute new state
     *         is ValueEvent.Destroy<Type>   -> // Compute new state
     *
     *         // These events only occur when observing a collection:
     *         is ValueEvent.Add<Type, *>    -> // Compute new state
     *         is ValueEvent.Remove<Type, *> -> // Compute new state
     *     }
     * }
     * ```
     *
     * The provided function must return the new computed value state which, unless it hasn't
     * changed, should be different ([equals]-wise) from the previous computed value state (as such,
     * when using an object as state, a new instance should be returned when the state is updated).
     * If the new state differs from the old one (using [equals]), the value will need to be
     * recomputed by the [form manager][FormManager] using [computeFromState].
     */
    protected fun <TValue> observe(
        pathToObserve: String,
        @BuilderInference updateState: UpdateStateFn<TValue, TState>,
    ): PropertyDelegateProvider<
        StatefulComputedValue<@UnsafeVariance T, TState>,
        ReadOnlyProperty<StatefulComputedValue<@UnsafeVariance T, TState>, Observer<TValue, TState>>,
    > = observe(Path(pathToObserve), updateState)

    /**
     * Initialises and returns the computed value's state, within a [ComputedValueContext]
     * containing the values of all declared dependencies.
     *
     * This method is called whenever a new value that needs to be computed by this computed value
     * is initialised.
     */
    public abstract override suspend fun ComputedValueContext.initState(): TState

    /**
     * Destroys the computed value's [state].
     *
     * Called whenever a state is no longer needed (i.e. when a newer state has been produced or
     * when the value being validated has been destroyed).
     */
    public override suspend fun destroyState(state: TState): Unit = Unit

    /**
     * Runs the computation, given its [state], within a [ComputedValueContext] containing the value
     * of all declared dependencies. Returns the computed value.
     *
     * This method is called instead of [compute] when computing stateful computed values.
     */
    public abstract suspend fun ComputedValueContext.computeFromState(state: TState): T

    // Default [compute] implementation for stateful computed values.
    override suspend fun ComputedValueContext.compute(): T = computeFromState(initState())
}
