package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.datatypes.File
import tech.ostack.kform.datatypes.Table
import tech.ostack.kform.validations.MaxSize.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.MinSize.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.Size.Companion.DEFAULT_CODE

/**
 * Validation that checks that a value's size is exactly [requiredSize], when it is not empty.
 * Values of type [Collection], [Array] (including variants), [Map], [Table], and [File] are
 * supported.
 *
 * When the value being validated is not empty, and its size is different from [requiredSize], then
 * an issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a
 * `size` data property with the size of the value that was validated and a `requiredSize` data
 * property with the value of [requiredSize].
 *
 * To also validate that a value is not empty, use [Required] (preferred) or [NotEmpty] (if the
 * value is nullable and `null` is an acceptable value) together with [Size].
 *
 * Example values accepted by schema `ListSchema(Size(4)) { IntSchema() }`:
 * - `listOf(1, 2, 3, 4)`
 * - `emptyList()`
 *
 * Example values rejected by schema `ListSchema(Size(4)) { IntSchema() }`:
 * - `listOf(1, 2, 3)`
 * - `listOf(1, 2, 3, 4, 5)`
 *
 * @property requiredSize Expected size of the value (must be >= 0).
 * @property code Issue code to use when the value is not empty and its size is different from
 *   [requiredSize].
 * @property severity Severity of the issue emitted when the value is not empty and its size is
 *   different from [requiredSize].
 */
public open class Size
@JvmOverloads
constructor(
    public val requiredSize: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<Any>() {
    init {
        require(requiredSize > 0) { "Provided size must be >= 0." }
    }

    override fun toString(): String = "Size($requiredSize)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val size = size(value)
        if (size != 0 && size != requiredSize) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("size" to "$size", "requiredSize" to "$requiredSize"),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the value's size is less than [requiredSize]. */
        public const val DEFAULT_CODE: String = "sizeMismatch"
    }
}

/**
 * Validation that checks that a value's size is at least a given [minSize], when it is not empty.
 * Values of type [Collection], [Array] (including variants), [Map], [Table], and [File] are
 * supported.
 *
 * When the value being validated is not empty and its size is less than [minSize], then an issue is
 * emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `size` data
 * property with the size of the value that was validated and a `minSize` data property with the
 * value of [minSize].
 *
 * To also validate that a value is not empty, use [Required] (preferred) or [NotEmpty] (if the
 * value is nullable and `null` is an acceptable value) together with [MinSize].
 *
 * Example values accepted by schema `ListSchema(MinSize(4)) { IntSchema() }`:
 * - `listOf(1, 2, 3, 4)`
 * - `listOf(1, 2, 3, 4, 5, 6, 7)`
 * - `emptyList()`
 *
 * Example values rejected by schema `ListSchema(MinSize(4)) { IntSchema() }`:
 * - `listOf(1)`
 * - `listOf(1, 2, 3)`
 *
 * @property minSize Minimum size allowed (must be >= 0).
 * @property code Issue code to use when the value is not empty and its size is less than [minSize].
 * @property severity Severity of the issue emitted when the value is not empty and its size is less
 *   than [minSize].
 */
public open class MinSize
@JvmOverloads
constructor(
    public val minSize: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<Any>() {
    init {
        require(minSize > 0) { "Provided minimum size must be >= 0." }
    }

    override fun toString(): String = "MinSize($minSize)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val size = size(value)
        if (size != 0 && size < minSize) {
            emit(ValidationIssue(code, severity, mapOf("size" to "$size", "minSize" to "$minSize")))
        }
    }

    public companion object {
        /** Default issue code representing that the value's size is less than [minSize]. */
        public const val DEFAULT_CODE: String = "tooSmall"
    }
}

/**
 * Validation that checks that a value's size is at most a given [maxSize]. Values of type
 * [Collection], [Array] (including variants), [Map], [Table], and [File] are supported.
 *
 * When the size of the value being validated is greater than [maxSize], then an issue is emitted
 * with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `size` data property
 * with the size of the value that was validated and a `maxSize` data property with the value of
 * [maxSize].
 *
 * Example values accepted by schema `ListSchema(MaxSize(4)) { IntSchema() }`:
 * - `listOf(1, 2)`
 * - `listOf(1, 2, 3, 4)`
 * - `emptyList()`
 *
 * Example values rejected by schema `ListSchema(MaxSize(4)) { IntSchema() }`:
 * - `listOf(1, 2, 3, 4, 5)`
 * - `listOf(1, 2, 3, 4, 5, 6, 7)`
 *
 * @property maxSize Maximum size allowed.
 * @property code Issue code to use when the value's size is greater than [maxSize].
 * @property severity Severity of the issue emitted when the value's size is greater than [maxSize].
 */
public open class MaxSize
@JvmOverloads
constructor(
    public val maxSize: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<Any>() {
    init {
        require(maxSize >= 0) { "Provided maximum size must be >= 0." }
    }

    override fun toString(): String = "MaxSize($maxSize)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val size = size(value)
        if (size > maxSize) {
            emit(ValidationIssue(code, severity, mapOf("size" to "$size", "maxSize" to "$maxSize")))
        }
    }

    public companion object {
        /** Default issue code representing that the value's size is greater than [maxSize]. */
        public const val DEFAULT_CODE: String = "tooLarge"
    }
}

/** Size of a [Collection], [Array] (including variants), [Map], [Table], or [File]. */
private fun size(value: Any): Int =
    when (value) {
        is Collection<*> -> value.size
        is Array<*> -> value.size
        is Map<*, *> -> value.size
        is Table<*> -> value.size
        is File -> value.size
        // Array variants
        is BooleanArray -> value.size
        is ByteArray -> value.size
        is CharArray -> value.size
        is DoubleArray -> value.size
        is FloatArray -> value.size
        is IntArray -> value.size
        is LongArray -> value.size
        is ShortArray -> value.size
        else ->
            error(
                "Unsupported value type: supported types are `Collection`, `Array` " +
                    "(including variants), `Map`, `Table`, and `File`."
            )
    }
