package tech.ostack.kform.validations

import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.emitAll
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.*
import tech.ostack.kform.validations.ValidationScopes.NotOneOf
import tech.ostack.kform.validations.ValidationScopes.OneOf

/**
 * Validation scopes under which a validation is allowed to run. These can be specified via an allow
 * list with [OneOf] or via a deny list with [NotOneOf].
 *
 * Note that, if no scope is provided as an external context during runtime, the default scope will
 * be `null`. In this scenario, for the allow list case, unless `null` is explicitely passed to
 * [OneOf], the validation will never run; similarly, for the deny list case, unless `null` is
 * explicitely passed to [NotOneOf], the validation will always run.
 */
public sealed class ValidationScopes<T>(scopes: Set<T?>) : Set<T?> by scopes {
    /** Allow list of scopes under which the validation should run. */
    public class OneOf<T>(scopes: Iterable<T?>) : ValidationScopes<T>(scopes.toSet()) {
        public constructor(vararg scopes: T?) : this(scopes.toSet())

        override fun allows(scope: T?): Boolean = scope in this
    }

    /** Deny list of scopes under which the validation should not run. */
    public class NotOneOf<T>(scopes: Iterable<T?>) : ValidationScopes<T>(scopes.toSet()) {
        public constructor(vararg scopes: T?) : this(scopes.toSet())

        override fun allows(scope: T?): Boolean = scope !in this
    }

    /** Whether the validation scopes allow the provided [scope]. */
    public abstract fun allows(scope: T?): Boolean
}

/**
 * A scoped validation is a validation that should only run when the current "validation scope" is
 * allowed by the provided [scopes].
 *
 * The current validation scope is obtained via an external context with a name defaulting to
 * [DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME] (which can be overriden during class construction).
 */
public abstract class ScopedValidation<in T, TScope>
@JvmOverloads
constructor(scopeExternalContextName: String = DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME) :
    Validation<T>() {
    /** Validation scopes of this validation. */
    public abstract val scopes: ValidationScopes<TScope>

    /** Current validation scope. */
    protected val ValidationContext.scope: TScope? by
        externalContextOrNull(scopeExternalContextName)

    public final override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (scopes.allows(scope)) {
            emitAll(scopedValidate())
        }
    }

    /**
     * Runs the scoped validation (when the current "validation scope" is allowed by the provided
     * [scopes]) within a [ValidationContext] containing the value being validated and the value of
     * all declared dependencies. Returns a flow over all found issues.
     */
    public abstract fun ValidationContext.scopedValidate(): Flow<ValidationIssue>

    public companion object {
        /** Default name of the external context from which to obtain the current scope. */
        public const val DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME: String = "scope"

        /**
         * Creates a new scoped validation from an already existing [validation] that runs when the
         * current "validation scope" is allowed by the provided [scopes].
         *
         * The current validation scope is obtained via an external context named
         * [scopeExternalContextName] (which defaults to [DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME]).
         */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun <T, TScope> invoke(
            validation: Validation<T>,
            scopes: ValidationScopes<TScope>,
            scopeExternalContextName: String = DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME,
        ): ScopedValidation<T, TScope> =
            WrappedScopedValidation(validation, scopes, scopeExternalContextName)

        /**
         * Creates a new scoped stateful validation from an already existing stateful [validation]
         * that runs when the current "validation scope" is allowed by the provided [scopes].
         *
         * The current validation scope is obtained via an external context named
         * [scopeExternalContextName] (which defaults to [DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME]).
         */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun <T, TState, TScope> invoke(
            validation: StatefulValidation<T, TState>,
            scopes: ValidationScopes<TScope>,
            scopeExternalContextName: String = DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME,
        ): ScopedStatefulValidation<T, TState, TScope> =
            WrappedScopedStatefulValidation(validation, scopes, scopeExternalContextName)
    }
}

/**
 * A scoped stateful validation is a stateful validation that should only run when the current
 * "validation scope" is allowed by the provided [scopes].
 *
 * The current validation scope is obtained via an external context with a name defaulting to
 * [ScopedValidation.DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME] (which can be overriden during class
 * construction).
 */
public abstract class ScopedStatefulValidation<in T, TState, TScope>
@JvmOverloads
constructor(
    scopeExternalContextName: String = ScopedValidation.DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME
) : StatefulValidation<T, TState>() {
    /** Validation scopes of this validation. */
    public abstract val scopes: ValidationScopes<TScope>

    /** Current validation scope. */
    protected val ValidationContext.scope: TScope? by
        externalContextOrNull(scopeExternalContextName)

    final override fun ValidationContext.validateFromState(state: TState): Flow<ValidationIssue> =
        flow {
            if (scopes.allows(scope)) {
                emitAll(scopedValidateFromState(state))
            }
        }

    final override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (scopes.allows(scope)) {
            emitAll(scopedValidateFromState(initState()))
        }
    }

    /**
     * Runs the scoped stateful validation (when the current "validation scope" is allowed by the
     * provided [scopes]), given its [state], within a [ValidationContext] containing the value
     * being validated and the value of all declared dependencies. Returns a flow over all found
     * issues.
     */
    public abstract fun ValidationContext.scopedValidateFromState(
        state: TState
    ): Flow<ValidationIssue>

    /**
     * Runs the scoped validation (when the current "validation scope" is allowed by the provided
     * [scopes]) within a [ValidationContext] containing the value being validated and the value of
     * all declared dependencies. Returns a flow over all found issues.
     */
    // Default [scopedValidate] implementation for scoped stateful validations.
    public open fun ValidationContext.scopedValidate(): Flow<ValidationIssue> = flow {
        emitAll(scopedValidateFromState(initState()))
    }
}

/** [Scoped validation][ScopedValidation] wrapping [validation]. */
private class WrappedScopedValidation<in T, TScope>(
    val validation: Validation<T>,
    override val scopes: ValidationScopes<TScope>,
    scopeExternalContextName: String = DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME,
) : ScopedValidation<T, TScope>(scopeExternalContextName) {
    override val dependencies: Map<String, DependencyInfo> = validation.dependencies
    override val dependsOnDescendants: Boolean = validation.dependsOnDescendants

    init {
        (externalContextDependencies as MutableSet<String>) +=
            validation.externalContextDependencies
    }

    override fun toString(): String = "Scoped$validation"

    override fun ValidationContext.scopedValidate(): Flow<ValidationIssue> =
        validation.run { validate() }
}

/** [Scoped stateful validation][ScopedStatefulValidation] wrapping [validation]. */
private class WrappedScopedStatefulValidation<in T, TState, TScope>(
    val validation: StatefulValidation<T, TState>,
    override val scopes: ValidationScopes<TScope>,
    scopeExternalContextName: String = ScopedValidation.DEFAULT_SCOPE_EXTERNAL_CONTEXT_NAME,
) : ScopedStatefulValidation<T, TState, TScope>(scopeExternalContextName) {
    override val dependencies: Map<String, DependencyInfo> = validation.dependencies
    override val dependsOnDescendants: Boolean = validation.dependsOnDescendants
    override val observers: List<Observer<Any?, TState>> = validation.observers

    init {
        (externalContextDependencies as MutableSet<String>) +=
            validation.externalContextDependencies
    }

    override fun toString(): String = "Scoped$validation"

    override suspend fun ValidationContext.initState(): TState = validation.run { initState() }

    override suspend fun destroyState(state: TState): Unit = validation.run { destroyState(state) }

    override fun ValidationContext.scopedValidate(): Flow<ValidationIssue> =
        validation.run { validate() }

    override fun ValidationContext.scopedValidateFromState(state: TState): Flow<ValidationIssue> =
        validation.run { validateFromState(state) }
}
