package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.datatypes.BigDecimal

/**
 * Validation that checks that a big decimal value's scale is [requiredScale].
 *
 * When the scale of the big decimal being validated is different from [requiredScale], then an
 * issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a
 * `value` data property with the value that was validated and a `requiredScale` data property with
 * the value of [requiredScale].
 *
 * @property requiredScale Required scale.
 * @property code Issue code to use when the value's scale is different from [requiredScale].
 * @property severity Severity of the issue emitted when the value's scale is different from
 *   [requiredScale].
 */
public open class Scale
@JvmOverloads
constructor(
    public val requiredScale: Int,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<BigDecimal>() {
    override fun toString(): String = "Scale($requiredScale)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value.scale != requiredScale) {
            println("value: $value, scale: ${value.scale}, required: $requiredScale")
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "requiredScale" to "$requiredScale"),
                )
            )
        }
    }

    public companion object {
        /**
         * Default issue code representing that the value's scale is different from [requiredScale].
         */
        public const val DEFAULT_CODE: String = "scaleMismatch"
    }
}
