package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.validations.ExclusiveMax.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.ExclusiveMin.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.Max.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.Min.Companion.DEFAULT_CODE

/**
 * Validation that checks that a value is at least a given [min] value.
 *
 * When the value being validated is less than [min], then an issue is emitted with the provided
 * [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property with the value
 * that was validated and a `min` data property with the value of [min].
 *
 * @property min Minimum value allowed.
 * @property code Issue code to use when the value is less than [min].
 * @property severity Severity of the issue emitted when the value is less than [min].
 */
public open class Min<T : Comparable<T>>
@JvmOverloads
constructor(
    public val min: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "Min($min)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value < min) {
            emit(ValidationIssue(code, severity, mapOf("value" to "$value", "min" to "$min")))
        }
    }

    public companion object {
        /** Default issue code representing that the value is less than [min]. */
        public const val DEFAULT_CODE: String = "rangeUnderflow"
    }
}

/**
 * Validation that checks that a value is at most a given [max] value.
 *
 * When the value being validated is greater than [max], then an issue is emitted with the provided
 * [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property with the value
 * that was validated and a `max` data property with the value of [max].
 *
 * @property max Maximum value allowed.
 * @property code Issue code to use when the value is greater than [max].
 * @property severity Severity of the issue emitted when the value is greater than [max].
 */
public open class Max<T : Comparable<T>>
@JvmOverloads
constructor(
    public val max: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "Max($max)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value > max) {
            emit(ValidationIssue(code, severity, mapOf("value" to "$value", "max" to "$max")))
        }
    }

    public companion object {
        /** Default issue code representing that the value is greater than [max]. */
        public const val DEFAULT_CODE: String = "rangeOverflow"
    }
}

/**
 * Validation that checks that a value is greater than a given [exclusiveMin] value.
 *
 * When the value being validated is less than or equal to [exclusiveMin], then an issue is emitted
 * with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data
 * property with the value that was validated and an `exclusiveMin` data property with the value of
 * [exclusiveMin].
 *
 * @property exclusiveMin Value that the value being validated must be greater than.
 * @property code Issue code to use when the value is less than or equal to [exclusiveMin].
 * @property severity Severity of the issue emitted when the value is less than or equal to
 *   [exclusiveMin].
 */
public open class ExclusiveMin<T : Comparable<T>>
@JvmOverloads
constructor(
    public val exclusiveMin: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "ExclusiveMin($exclusiveMin)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value <= exclusiveMin) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "exclusiveMin" to "$exclusiveMin"),
                )
            )
        }
    }

    public companion object {
        /**
         * Default issue code representing that the value is less than or equal to [exclusiveMin].
         */
        public const val DEFAULT_CODE: String = "rangeUnderflow"
    }
}

/**
 * Validation that checks that a value is less than a given [exclusiveMax] value.
 *
 * When the value being validated is greater than or equal to [exclusiveMax], then an issue is
 * emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data
 * property with the value that was validated and an `exclusiveMax` data property with the value of
 * [exclusiveMax].
 *
 * @property exclusiveMax Value that the value being validated must be less than.
 * @property code Issue code to use when the value is greater than or equal to [exclusiveMax].
 * @property severity Severity of the issue emitted when the value is greater than or equal to
 *   [exclusiveMax].
 */
public open class ExclusiveMax<T : Comparable<T>>
@JvmOverloads
constructor(
    public val exclusiveMax: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "ExclusiveMax($exclusiveMax)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value >= exclusiveMax) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "exclusiveMax" to "$exclusiveMax"),
                )
            )
        }
    }

    public companion object {
        /**
         * Default issue code representing that the value is greater than or equal to
         * [exclusiveMax].
         */
        public const val DEFAULT_CODE: String = "rangeOverflow"
    }
}
