package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.validations.MustEqual.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.NotOneOf.Companion.DEFAULT_CODE
import tech.ostack.kform.validations.OneOf.Companion.DEFAULT_CODE

/**
 * Validation that checks that a value equals a given [requiredValue].
 *
 * When the value being validated is different from [requiredValue], then an issue is emitted with
 * the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property
 * with the value that was validated and a `requiredValue` data property with the value of
 * [requiredValue].
 *
 * @property requiredValue Value to equal.
 * @property code Issue code to use when the value is different from [requiredValue].
 * @property severity Severity of the issue emitted when the value is different from
 *   [requiredValue].
 */
public open class MustEqual<T>
@JvmOverloads
constructor(
    public val requiredValue: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "MustEqual($requiredValue)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value != requiredValue) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "requiredValue" to "$requiredValue"),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the value is different from [requiredValue]. */
        public const val DEFAULT_CODE: String = "valueDisallowed"
    }
}

/**
 * Validation that checks that a value does not equal a given [forbiddenValue].
 *
 * When the value being validated is equal to [forbiddenValue], then an issue is emitted with the
 * provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property with
 * the value that was validated and a `forbiddenValue` data property with the value of
 * [forbiddenValue].
 *
 * @property forbiddenValue Value to differ from.
 * @property code Issue code to use when the value is equal to [forbiddenValue].
 * @property severity Severity of the issue emitted when the value is equal to [forbiddenValue].
 */
public open class MustNotEqual<T>
@JvmOverloads
constructor(
    public val forbiddenValue: T,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    override fun toString(): String = "MustNotEqual($forbiddenValue)"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value == forbiddenValue) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "forbiddenValue" to "$forbiddenValue"),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the value is equal to [forbiddenValue]. */
        public const val DEFAULT_CODE: String = "valueDisallowed"
    }
}

/**
 * Validation that checks that a value is one of the [allowedValues].
 *
 * When the value being validated is not one of the [allowedValues], then an issue is emitted with
 * the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property
 * with the value that was validated and an `allowedValues` data property with the [allowedValues].
 *
 * @property allowedValues Allowed values.
 * @property code Issue code to use when the value is not one of the allowed values.
 * @property severity Severity of the issue emitted when the value is not one of the allowed values.
 */
public open class OneOf<T>
@JvmOverloads
constructor(
    allowedValues: Iterable<T>,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    init {
        require(allowedValues.any()) { "At least one allowed value must be provided." }
    }

    public val allowedValues: Set<T> = allowedValues.toSet()

    @JvmOverloads
    public constructor(
        vararg allowedValues: T,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(allowedValues.asIterable(), code, severity)

    override fun toString(): String = "OneOf(${allowedValues.joinToString(", ")})"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value !in allowedValues) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf("value" to "$value", "allowedValues" to allowedValues.joinToString(", ")),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the value is not one of the [allowedValues]. */
        public const val DEFAULT_CODE: String = "valueDisallowed"
    }
}

/**
 * Validation that checks that a value is **not** one of the [disallowedValues].
 *
 * When the value being validated is one of the [disallowedValues], then an issue is emitted with
 * the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a `value` data property
 * with the value that was validated and a `disallowedValues` data property with the
 * [disallowedValues].
 *
 * @property disallowedValues Disallowed values.
 * @property code Issue code to use when the value is one of the disallowed values.
 * @property severity Severity of the issue emitted when the value is one of the disallowed values.
 */
public open class NotOneOf<T>
@JvmOverloads
constructor(
    disallowedValues: Iterable<T>,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<T>() {
    init {
        require(disallowedValues.any()) { "At least one disallowed value must be provided." }
    }

    public val disallowedValues: Set<T> = disallowedValues.toSet()

    @JvmOverloads
    public constructor(
        vararg disallowedValues: T,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(disallowedValues.asIterable(), code, severity)

    override fun toString(): String = "NotOneOf(${disallowedValues.joinToString(", ")})"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        if (value in disallowedValues) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf(
                        "value" to "$value",
                        "disallowedValues" to disallowedValues.joinToString(", "),
                    ),
                )
            )
        }
    }

    public companion object {
        /** Default issue code representing that the value is one of the [disallowedValues]. */
        public const val DEFAULT_CODE: String = "valueDisallowed"
    }
}
