package tech.ostack.kform.validations

import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import tech.ostack.kform.Validation
import tech.ostack.kform.ValidationContext
import tech.ostack.kform.ValidationIssue
import tech.ostack.kform.ValidationIssueSeverity
import tech.ostack.kform.datatypes.File
import tech.ostack.kform.validations.Accepts.Companion.DEFAULT_CODE

/** Representation of a MIME type as a pair of type/subtype. */
private typealias MimeType = Pair<String, String>

/**
 * Validation that checks that a [file][File]'s type is one of the [acceptedFileTypes].
 *
 * Each provided file type to accept must match one of the following:
 * - A MIME type, indicating that files of the specified type are accepted (provided MIME types may
 *   contain wildcards such as `image/∗`).
 * - A string whose first character is a full stop (`.`), indicating that files with the specified
 *   file extension are accepted.
 *
 * When the type of the file being validated does not match one of the [acceptedFileTypes], then an
 * issue is emitted with the provided [code] (defaults to [DEFAULT_CODE]). This issue contains a
 * `fileName` data property with the name of the file that was validated, a `mimeType` data property
 * with the MIME type of the file that was validated, and an `accepted` data property with the
 * [acceptedFileTypes].
 *
 * Example usage:
 * ```
 * Accepts("image/∗", ".pdf")
 * ```
 *
 * @property acceptedFileTypes Accepted file types (each accepted file type must be either a MIME
 *   type or a file extension starting with a full stop).
 * @property code Issue code to use when the file's type does not match one of the accepted file
 *   types.
 * @property severity Severity of the issue emitted when the file's type does not match one of the
 *   accepted file types.
 */
public open class Accepts
@JvmOverloads
constructor(
    acceptedFileTypes: Iterable<String>,
    public val code: String = DEFAULT_CODE,
    public val severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
) : Validation<File>() {
    init {
        require(acceptedFileTypes.any()) {
            "At least one type of file to be accepted must be provided."
        }
    }

    private val acceptedMimeTypes: MutableSet<MimeType> = mutableSetOf()
    private val acceptedFileExtensions: MutableSet<String> = mutableSetOf()

    init {
        for (type in acceptedFileTypes) {
            if (type.startsWith('.')) {
                acceptedFileExtensions += type.lowercase()
            } else {
                acceptedMimeTypes +=
                    type.toMimeTypeOrNull()
                        ?: throw IllegalArgumentException("Invalid file type: '$type'.")
            }
        }
    }

    public val acceptedFileTypes: Set<String>
        get() = buildSet {
            addAll(acceptedMimeTypes.map { it.toMimeTypeString() })
            addAll(acceptedFileExtensions)
        }

    @JvmOverloads
    public constructor(
        vararg acceptedFileTypes: String,
        code: String = DEFAULT_CODE,
        severity: ValidationIssueSeverity = ValidationIssueSeverity.Error,
    ) : this(acceptedFileTypes.asIterable(), code, severity)

    override fun toString(): String = "Accepts(${acceptedFileTypes.joinToString(", ")})"

    override fun ValidationContext.validate(): Flow<ValidationIssue> = flow {
        val fileExtension = getFileExtension(value.name)
        val fileMimeType = value.type?.toMimeTypeOrNull()
        if (
            (fileExtension == null || acceptedFileExtensions.none { fileExtension.endsWith(it) }) &&
                (fileMimeType == null ||
                    acceptedMimeTypes.none { mimeTypesMatch(fileMimeType, it) })
        ) {
            emit(
                ValidationIssue(
                    code,
                    severity,
                    mapOf(
                        "fileName" to value.name,
                        "mimeType" to fileMimeType?.toMimeTypeString(),
                        "acceptedFileTypes" to acceptedFileTypes.joinToString(", "),
                    ),
                )
            )
        }
    }

    /** Gets the file extension of a file (including the leading full stop). */
    private fun getFileExtension(fileName: String): String? {
        val fullStopIdx = fileName.indexOf('.')
        return if (fullStopIdx == -1) null else fileName.drop(fullStopIdx).lowercase()
    }

    /** Transforms a string into a MIME type (represented as a pair). */
    private fun String.toMimeTypeOrNull(): MimeType? {
        val slashIdx = indexOf('/')
        return if (slashIdx == -1) null
        else MimeType(take(slashIdx).lowercase(), drop(slashIdx + 1).lowercase())
    }

    /** Transforms a MIME type (represented as a pair) into a string. */
    private fun MimeType.toMimeTypeString(): String = "$first/$second"

    /** Whether the file's MIME type matches the provided accepted MIME type. */
    private fun mimeTypesMatch(fileMimeType: MimeType, acceptedMimeType: MimeType): Boolean {
        val (fileType, fileSubtype) = fileMimeType
        val (acceptedType, acceptedSubtype) = acceptedMimeType
        return (acceptedType == "*" || fileType == acceptedType) &&
            (acceptedSubtype == "*" || fileSubtype == acceptedSubtype)
    }

    public companion object {
        /**
         * Default issue code representing that the file's type does not match one of the
         * [acceptedFileTypes] file types.
         */
        public const val DEFAULT_CODE: String = "fileTypeMismatch"
    }
}
