@file:JvmName("SchemaPathUtils")

package tech.ostack.kform.util

import kotlin.jvm.JvmName
import kotlin.math.min
import tech.ostack.kform.*

/**
 * Compares two paths of a schema according to the order in which children are defined in said
 * schema. Returns a value `> 0` if [path1] is greater than [path2], `< 0` if [path2] if greater
 * than [path1], or `0` otherwise.
 *
 * Consider the following classes and schema:
 * ```kotlin
 * data class Person(var name: String, var pets: List<Pet>)
 * data class Pet(var name: String, var age: Int)
 *
 * val formSchema = ClassSchema {
 *     Person::name { StringSchema() }
 *     Person::pets {
 *         ListSchema {
 *             ClassSchema {
 *                 Pet::name { StringSchema() }
 *                 Pet::age { IntSchema() }
 *             }
 *         }
 *     }
 * }
 * ```
 *
 * For the above schema, all the following are true:
 * - `/name` &gt; `/`
 * - `/pets` &gt; `/name`
 * - `/pets/1` &gt; `/pets/0`
 * - `/pets/1/name` &gt; `/pets/1`
 * - `/pets/1/age` &gt; `/pets/1/name`
 */
public fun Schema<*>.comparePaths(path1: Path, path2: Path): Int {
    val absolutePath1 = path1.toAbsolutePath()
    val absolutePath2 = path2.toAbsolutePath()

    var curSchema = this
    for (i in 0 until min(absolutePath1.size, absolutePath2.size)) {
        val id1 =
            absolutePath1[i] as? AbsolutePathFragment.Id
                ?: throw InvalidPathException(
                    absolutePath1,
                    "Compared paths must only contain ids.",
                )
        val id2 =
            absolutePath2[i] as? AbsolutePathFragment.Id
                ?: throw InvalidPathException(
                    absolutePath2,
                    "Compared paths must only contain ids.",
                )
        when (curSchema) {
            is CollectionSchema<*, *> -> {
                if (id1 != id2) {
                    val int1 = id1.id.toPositiveIntWithoutLeadingZeroesOrNull()
                    val int2 = id2.id.toPositiveIntWithoutLeadingZeroesOrNull()
                    return if (int1 != null && int2 != null) int1.compareTo(int2)
                    else id1.id.compareTo(id2.id)
                }
                curSchema = schema(curSchema, Path.CHILDREN)
            }
            is ParentSchema<*> -> {
                if (id1 != id2) {
                    val childrenSchemasInfo =
                        curSchema.childrenSchemas(
                            AbsolutePath.ROOT,
                            AbsolutePath.ROOT,
                            AbsolutePathFragment.Wildcard,
                        )
                    val id1Index = childrenSchemasInfo.indexOfFirst { it.path.lastFragment == id1 }
                    if (id1Index == -1) {
                        throw InvalidPathException(
                            AbsolutePath(absolutePath1.fragments.slice(0..i)),
                            "No schema matches this path.",
                        )
                    }
                    val id2Index = childrenSchemasInfo.indexOfFirst { it.path.lastFragment == id2 }
                    if (id2Index == -1) {
                        throw InvalidPathException(
                            AbsolutePath(absolutePath1.fragments.slice(0..i)),
                            "No schema matches this path.",
                        )
                    }
                    return id1Index.compareTo(id2Index)
                }
                curSchema =
                    curSchema
                        .childrenSchemas(AbsolutePath.ROOT, AbsolutePath.ROOT, id1)
                        .single()
                        .schema
            }
            else ->
                throw InvalidPathException(
                    AbsolutePath(absolutePath1.fragments.slice(0 until i)),
                    "Schema is not a parent schema.",
                )
        }
    }
    return absolutePath1.size.compareTo(absolutePath2.size)
}

/**
 * Compares two paths of a schema according to the order in which children are defined in said
 * schema. Returns a value `> 0` if [path1] is greater than [path2], `< 0` if [path2] if greater
 * than [path1], or `0` otherwise.
 *
 * Consider the following classes and schema:
 * ```kotlin
 * data class Person(var name: String, var pets: List<Pet>)
 * data class Pet(var name: String, var age: Int)
 *
 * val formSchema = ClassSchema {
 *     Person::name { StringSchema() }
 *     Person::pets {
 *         ListSchema {
 *             ClassSchema {
 *                 Pet::name { StringSchema() }
 *                 Pet::age { IntSchema() }
 *             }
 *         }
 *     }
 * }
 * ```
 *
 * For the above schema, all the following are true:
 * - `/name` &gt; `/`
 * - `/pets` &gt; `/name`
 * - `/pets/1` &gt; `/pets/0`
 * - `/pets/1/name` &gt; `/pets/1`
 * - `/pets/1/age` &gt; `/pets/1/name`
 */
public fun Schema<*>.comparePaths(path1: String, path2: String): Int =
    comparePaths(AbsolutePath(path1), AbsolutePath(path2))

// `toIntOrNull` implementation where strings with leading zeroes or representing negative values
// return `null`
private fun String.toPositiveIntWithoutLeadingZeroesOrNull(): Int? =
    if (startsWith("-") || (startsWith("0") && length != 1)) null else toIntOrNull()
