package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.isActive
import tech.ostack.kform.*
import tech.ostack.kform.datatypes.*
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.sizeBoundsRestrictions

/** Schema representing a table of values of type [T] with a schema of type [TSchema]. */
public open class TableSchema<T, TSchema : Schema<T>>(
    elementsSchema: TSchema,
    validations: Iterable<Validation<Table<T>>> = emptyList(),
    override val initialValue: Table<T> = tableOf(),
) : AbstractCollectionSchema<Table<T>, T, TSchema>(elementsSchema) {
    override val typeInfo: TypeInfo =
        TypeInfo(
            Table::class,
            arguments = listOf(elementsSchema.typeInfo),
            restrictions = commonRestrictions(validations) + sizeBoundsRestrictions(validations),
        )

    override val validations: List<Validation<Table<T>>> = validations.toList()

    public constructor(
        validations: List<Validation<Table<T>>> = emptyList(),
        initialValue: Table<T> = tableOf(),
        @BuilderInference builder: () -> TSchema,
    ) : this(builder(), validations, initialValue)

    public constructor(
        vararg validations: Validation<Table<T>>,
        initialValue: Table<T> = tableOf(),
        @BuilderInference builder: () -> TSchema,
    ) : this(builder(), validations.asIterable(), initialValue)

    override suspend fun clone(value: Table<T>): Table<T> {
        val table = Table<T>(value.size)
        for ((id, rowValue) in value.rows) {
            table[id] = elementsSchema.clone(rowValue)
        }
        return table
    }

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(initialValue) == true &&
            (if (type.classifier == Table::class)
                type.arguments[0].type == null ||
                    elementsSchema.assignableTo(type.arguments[0].type!!)
            else true)

    /** Whether the provided id fragment can be converted to a table row id. */
    private fun isValidRowId(fragment: AbsolutePathFragment.Id): Boolean =
        fragment.id.matches(ROW_ID_REGEX)

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id && isValidRowId(fragment))

    override suspend fun isValidChildFragment(
        value: Table<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidRowId(fragment) &&
            fragment.id.toInt() in value)

    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: Table<T>,
        fragment: AbsolutePathFragment,
    ): Flow<ValueInfo<T>> = flow {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((id, elem) in value.rows) {
                emit(
                    ValueInfo(
                        elem,
                        elementsSchema,
                        path.append(AbsolutePathFragment.Id(id)),
                        schemaPath.append(AbsolutePathFragment.Wildcard),
                    )
                )
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val id = fragment.id.toInt()
            if (id in value) {
                emit(
                    ValueInfo(
                        value[id]!!,
                        elementsSchema,
                        path.append(fragment),
                        schemaPath.append(AbsolutePathFragment.Wildcard),
                    )
                )
            }
        }
    }

    @Suppress("UNCHECKED_CAST")
    private fun fromAny(value: Any?): Table<T> =
        when (value) {
            is Table<*> -> value as Table<T>
            is Map<*, *> -> (value as Map<TableRowId, T>).toTable()
            is Collection<*> -> (value as Collection<T>).toTable()
            is Array<*> -> (value as Array<T>).toTable()
            else -> throw IllegalArgumentException("Cannot convert value '$value' to Table.")
        }

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: Table<T>) -> Unit,
    ) {
        val fromTable = fromAny(fromValue)
        val newValue = Table<T>(fromTable.size)
        for ((id, elem) in fromTable.rows) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            elementsSchema.init(path.append(AbsolutePathFragment.Id(id)), elem, eventsBus) {
                newValue[id] = it
            }
        }
        setValue(newValue)
        eventsBus.emit(ValueEvent.Init(newValue, path, this))
    }

    override suspend fun change(
        path: AbsolutePath,
        value: Table<T>,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: Table<T>) -> Unit,
    ) {
        val intoTable = fromAny(intoValue)
        val newValue = Table<T>(intoTable.size)
        for ((id, elem) in intoTable.rows) {
            val elPath = path.append(AbsolutePathFragment.Id(id))
            if (id in value)
                elementsSchema.change(elPath, value[id]!!, elem, eventsBus) { newValue[id] = it }
            else elementsSchema.init(elPath, elem, eventsBus) { newValue[id] = it }
        }
        for ((id, elem) in value.rows) {
            if (id !in intoTable) {
                elementsSchema.destroy(path.append(AbsolutePathFragment.Id(id)), elem, eventsBus) {}
            }
        }
        setValue(newValue)
        eventsBus.emit(ValueEvent.Change(value, newValue, path, this))
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: Table<T>,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (value: Table<T>) -> Unit,
    ) {
        removeValue(value)
        eventsBus.emit(ValueEvent.Destroy(value, path, this))
        for (i in value.size - 1 downTo 0) {
            val row = value.rowAt(i)
            elementsSchema.destroy(
                path.append(AbsolutePathFragment.Id(row.id)),
                row.value,
                eventsBus,
            ) {}
        }
    }

    override suspend fun isValidSetFragment(
        value: Table<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        fragment is AbsolutePathFragment.CollectionEnd ||
            (fragment is AbsolutePathFragment.Id && isValidRowId(fragment))

    override suspend fun set(
        path: AbsolutePath,
        value: Table<T>,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus,
    ) {
        if (fragment == AbsolutePathFragment.Wildcard) {
            for ((id, elem) in value.rows) {
                elementsSchema.change(
                    path.append(AbsolutePathFragment.Id(id)),
                    elem,
                    childValue,
                    eventsBus,
                ) {
                    value[id] = it
                }
            }
        }
        val idFragment: AbsolutePathFragment.Id =
            if (fragment is AbsolutePathFragment.CollectionEnd)
                AbsolutePathFragment.Id(value.nextId)
            else fragment as AbsolutePathFragment.Id
        val id = idFragment.id.toInt()
        if (id !in value) {
            elementsSchema.init(path.append(idFragment), childValue, eventsBus) { value[id] = it }
            eventsBus.emit(ValueEvent.Add(value, value[id]!!, idFragment, path, this))
        } else {
            elementsSchema.change(path.append(idFragment), value[id]!!, childValue, eventsBus) {
                value[id] = it
            }
        }
    }

    override suspend fun isValidRemoveFragment(
        value: Table<T>,
        fragment: AbsolutePathFragment,
    ): Boolean =
        (fragment is AbsolutePathFragment.Id &&
            isValidRowId(fragment) &&
            fragment.id.toInt() in value)

    override suspend fun remove(
        path: AbsolutePath,
        value: Table<T>,
        fragment: AbsolutePathFragment,
        eventsBus: SchemaEventsBus,
    ) {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for (i in value.size - 1 downTo 0) {
                val (id, elem) = value.rowAt(i)
                val idFragment = AbsolutePathFragment.Id(id)
                elementsSchema.destroy(path.append(idFragment), elem, eventsBus) {
                    value.removeAt(i)
                }
                eventsBus.emit(ValueEvent.Remove(value, elem, idFragment, path, this))
            }
        } else {
            val id = (fragment as AbsolutePathFragment.Id).id.toInt()
            val elem = value[id]!!
            elementsSchema.destroy(path.append(fragment), elem, eventsBus) { value -= id }
            eventsBus.emit(ValueEvent.Remove(value, elem, fragment, path, this))
        }
    }

    override fun childrenStatesContainer(): CollectionState = TableState(elementsSchema)

    public companion object {
        /**
         * Regex used to determine if a certain identifier can be converted to a table row
         * identifier.
         */
        private val ROW_ID_REGEX = Regex("^(0|-?[1-9]\\d*)$")
    }
}

/** Class responsible for holding the states of the children of a list. */
public class TableState(private val elementsSchema: Schema<*>) : CollectionState {
    /** List containing the state of each element of the list. */
    private val childrenStatesMap = hashMapOf<TableRowId, State?>()

    override fun childrenStates(
        path: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<StateInfo<*>> = sequence {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((id, state) in childrenStatesMap) {
                yield(StateInfo(state, elementsSchema, path + AbsolutePathFragment.Id(id)))
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val id = fragment.id.toInt()
            if (id in childrenStatesMap) {
                yield(StateInfo(childrenStatesMap[id], elementsSchema, path + fragment))
            }
        }
    }

    override fun hasState(fragment: AbsolutePathFragment.Id): Boolean =
        fragment.id.toInt() in childrenStatesMap

    override fun setState(fragment: AbsolutePathFragment.Id, state: State?) {
        childrenStatesMap[fragment.id.toInt()] = state
    }

    override fun removeState(fragment: AbsolutePathFragment.Id) {
        childrenStatesMap.remove(fragment.id.toInt())
    }
}
