package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions

/** Schema representing numeric values of type [Long]. */
public open class LongSchema(
    validations: Iterable<Validation<Long>> = emptyList(),
    override val initialValue: Long = 0L,
) : AbstractSimpleSchema<Long>(validations) {
    public constructor(
        vararg validations: Validation<Long>,
        initialValue: Long = 0L,
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            Long::class,
            restrictions =
                commonRestrictions(validations) +
                    comparableBoundsRestrictions(validations, Long.MIN_VALUE, Long.MAX_VALUE),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(0L) == true

    override suspend fun fromAny(value: Any?): Long =
        when (value) {
            is Number -> value.toLong()
            is String -> value.toLong()
            else -> throw IllegalArgumentException("Cannot convert value '$value' to Long.")
        }
}
