package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import kotlin.time.Instant
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions

/** Schema representing values of type [Instant]. */
public open class InstantSchema(
    validations: Iterable<Validation<Instant>> = emptyList(),
    override val initialValue: Instant = DEFAULT_INITIAL_VALUE,
) : AbstractSimpleSchema<Instant>(validations) {
    public constructor(
        vararg validations: Validation<Instant>,
        initialValue: Instant = DEFAULT_INITIAL_VALUE,
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            Instant::class,
            restrictions =
                commonRestrictions(validations) + comparableBoundsRestrictions(validations),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(DEFAULT_INITIAL_VALUE) == true

    override suspend fun fromAny(value: Any?): Instant =
        when (value) {
            is Instant -> value
            is Number -> Instant.fromEpochMilliseconds(value.toLong())
            is String -> Instant.parse(value)
            else -> throw IllegalArgumentException("Cannot convert value '$value' to Instant.")
        }

    public companion object {
        /** Default initial value (epoch). */
        public val DEFAULT_INITIAL_VALUE: Instant = Instant.fromEpochMilliseconds(0)
    }
}
