package tech.ostack.kform.schemas

import kotlin.reflect.KClass
import kotlin.reflect.KType
import org.gciatto.kt.math.RoundingMode
import tech.ostack.kform.TypeInfo
import tech.ostack.kform.Validation
import tech.ostack.kform.datatypes.BigDecimal
import tech.ostack.kform.datatypes.BigInteger
import tech.ostack.kform.schemas.util.commonRestrictions
import tech.ostack.kform.schemas.util.comparableBoundsRestrictions
import tech.ostack.kform.schemas.util.scaleRestriction
import tech.ostack.kform.validations.Scale

/** Schema representing numeric values of type [BigDecimal]. */
public open class BigDecimalSchema(
    validations: Iterable<Validation<BigDecimal>> = emptyList(),
    override val initialValue: BigDecimal = zeroOfPreferredScale(validations),
) : AbstractSimpleSchema<BigDecimal>(validations) {
    public val preferredScale: Int? = preferredScale(validations)

    public constructor(
        vararg validations: Validation<BigDecimal>,
        initialValue: BigDecimal = zeroOfPreferredScale(validations.asIterable()),
    ) : this(validations.asIterable(), initialValue)

    override val typeInfo: TypeInfo =
        TypeInfo(
            BigDecimal::class,
            restrictions =
                commonRestrictions(validations) +
                    comparableBoundsRestrictions(validations) +
                    scaleRestriction(validations),
        )

    override fun assignableTo(type: KType): Boolean =
        (type.classifier as? KClass<*>)?.isInstance(BigDecimal.ZERO) == true

    override suspend fun fromAny(value: Any?): BigDecimal {
        val bigDecimal =
            when (value) {
                is BigDecimal -> return value
                is BigInteger -> BigDecimal.of(value)
                is Byte -> BigDecimal.of(value.toInt())
                is Short -> BigDecimal.of(value.toInt())
                is Int -> BigDecimal.of(value)
                is Long -> BigDecimal.of(value)
                is Float -> BigDecimal.of(value)
                is Double -> BigDecimal.of(value)
                is String -> BigDecimal.of(value)
                else ->
                    throw IllegalArgumentException("Cannot convert value '$value' to BigDecimal.")
            }
        return if (preferredScale == null || bigDecimal.scale >= preferredScale) bigDecimal
        else bigDecimal.setScale(preferredScale, RoundingMode.UNNECESSARY)
    }

    internal companion object {
        /**
         * Returns the preferred scale of the big decimal schema inferred from the provided
         * validations.
         */
        private fun preferredScale(validations: Iterable<Validation<BigDecimal>>): Int? {
            val requiredScales =
                validations.filterIsInstance<Scale>().map { it.requiredScale }.toSet()
            return if (requiredScales.size == 1) requiredScales.first() else null
        }

        /**
         * Returns a big decimal representation of zero with the preferred scale inferred from the
         * provided validations.
         */
        internal fun zeroOfPreferredScale(
            validations: Iterable<Validation<BigDecimal>>
        ): BigDecimal {
            val preferredScale = preferredScale(validations)
            return if (preferredScale == null) BigDecimal.ZERO
            else BigDecimal.ZERO.setScale(preferredScale)
        }
    }
}
