package tech.ostack.kform.internal.actions

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.onCompletion
import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/**
 * Action that calls [issuesHandler] with a flow over the validation issues found when validating
 * all values matching [path].
 */
internal class ValidateAction<T>(
    formManager: FormManager,
    private val path: AbsolutePath,
    private val issuesHandler: IssuesHandler<T>,
    override val priority: Int = 0,
) : ValueStateAction<T>(formManager) {
    override fun toString() = "Validate($path)"

    override val accesses =
        mutableListOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Write),
            AccessStatefulValidationDeferredState(ActionAccessType.Read),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )

    override val accessedPaths: Set<AbsolutePath>

    init {
        // Iterate over all validations that need to run and determine the accessed paths and
        // whether access to external context is required
        var requiresAccessToExternalContext = false
        val paths = mutableSetOf<AbsolutePath>()
        for ((schema, _, queriedPath) in schemaInfo(path)) {
            val validations = schema.validations
            if (validations.isNotEmpty()) {
                paths += queriedPath + AbsolutePathFragment.RecursiveWildcard
            }
            for (validation in validations) {
                for (dependency in validation.dependencies.values) {
                    paths +=
                        queriedPath.resolve(dependency.path) +
                            AbsolutePathFragment.RecursiveWildcard
                }
                if (validation.externalContextDependencies.isNotEmpty()) {
                    requiresAccessToExternalContext = true
                }
            }
        }

        if (requiresAccessToExternalContext) {
            accesses += AccessExternalContext(ActionAccessType.Read)
        }
        accessedPaths = paths
    }

    override suspend fun runValueState(): T = issuesHandler(validate())

    /** Validates all values matching the action's path. */
    // TODO: Possibly run validations with some concurrency
    private fun validate(): Flow<LocatedValidationIssue> = flow {
        valueStateInfo(path).collect { (value, state, schema, path, schemaPath) ->
            state as StateImpl

            // Old statuses
            val oldLocalDisplayStatus = state.localDisplayStatus()
            val oldDisplayStatus = state.displayStatus()

            // When not already (successfully) validated, set status as "validating"
            val wasSuccessfullyValidated = state.validationStatus === ValidationStatus.Validated
            if (!wasSuccessfullyValidated) {
                state.validationStatus = ValidationStatus.Validating
                formManager.eventsBus.emit(
                    StateEvent.ValidationChange(
                        ValidationStatus.Validating,
                        state.getAllVisibleIssues(),
                        path,
                        schema,
                    )
                )
            }
            var validationCancelled = false
            var validationFailed = false

            // Keep track of emitted issues in order to prevent emitting repeated ones
            val emittedIssues = mutableSetOf<LocatedValidationIssue>()

            var statefulValidationIndex = 0
            for ((validationIndex, validation) in schema.validations.withIndex()) {
                val issues = state.getCachedIssues(validationIndex)
                // Rerun the validation if the previous run failed (validation threw an exception)
                if (issues != null && !issues.any { it is ValidationFailure }) {
                    FormManager.logger.trace {
                        "At '$path': Emitting cached issues for validation '$validation'"
                    }
                    // Emit cached issues
                    for (issue in issues) {
                        val locatedIssue = LocatedValidationIssue(path, validation, issue)
                        if (emittedIssues.add(locatedIssue)) {
                            emit(locatedIssue)
                        }
                    }
                } else {
                    runValidation(
                            value,
                            schema,
                            path,
                            schemaPath,
                            state,
                            validation,
                            validationIndex,
                            statefulValidationIndex,
                        )
                        .onCompletion { ex ->
                            if (ex is CancellationException) {
                                validationCancelled = true
                            }
                        }
                        .collect { issue ->
                            if (issue is ValidationFailure) {
                                validationFailed = true
                            }
                            val locatedIssue = LocatedValidationIssue(path, validation, issue)
                            if (emittedIssues.add(locatedIssue)) {
                                emit(locatedIssue)
                            }
                        }
                }

                if (validation is StatefulValidation<*, *>) {
                    ++statefulValidationIndex
                }
            }

            // Emit external issues
            for (externalIssue in state.externalIssues ?: emptyArray()) {
                if (emittedIssues.add(externalIssue)) {
                    emit(externalIssue)
                }
            }

            // When it wasn't already (successfully) validated, update the statuses
            if (!wasSuccessfullyValidated) {
                state.validationStatus =
                    when {
                        validationCancelled -> ValidationStatus.Unvalidated
                        validationFailed -> ValidationStatus.ValidatedExceptionally
                        else -> ValidationStatus.Validated
                    }
            }
            updateValidationState(
                formManager,
                schema,
                path,
                state,
                oldLocalDisplayStatus,
                oldDisplayStatus,
                descendantsDisplayingIssuesToUpdate,
                emitValidationChange = !wasSuccessfullyValidated,
            )
        }
    }

    /**
     * Runs a single validation and returns a flow over its validation issues. This function may
     * throw a `CancellationException`. The [statefulValidationIndex] is unused when the validation
     * isn't stateful, so any number can be provided.
     */
    private fun runValidation(
        value: Any?,
        schema: Schema<*>,
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        state: StateImpl,
        validation: Validation<*>,
        validationIndex: Int,
        statefulValidationIndex: Int,
    ): Flow<ValidationIssue> = flow {
        val validationContext =
            ValidationContext(
                value,
                schema,
                path,
                schemaPath,
                dependenciesInfo(path, validation.dependencies),
                validation.externalContextDependencies.associateWith {
                    formManager.externalContexts[it]
                },
            )

        // Wrap code in `try/catch` since we're calling user code (`validate` or
        // `validateFromState`) and an error may occur when **creating** the flow; an error may also
        // occur while initialising or accessing the state of a stateful validation
        val issuesFlow: Flow<ValidationIssue>?
        try {
            @Suppress("UNCHECKED_CAST")
            issuesFlow =
                if (validation is StatefulValidation<*, *>) {
                    validation as StatefulValidation<Any?, Any?>
                    var deferredState =
                        state.getStatefulValidationDeferredState(statefulValidationIndex)
                    // Initialise validation state if it has not yet been initialised; this may fail
                    // due to an error in the `initState` function or due to cancellation
                    if (deferredState == null) {
                        deferredState =
                            initStatefulValidationState(
                                path,
                                state,
                                validation,
                                statefulValidationIndex,
                                validationContext,
                            )
                    }
                    // Wait for the validation state in case ongoing state updates exist; this may
                    // fail due to an error during the state update, in which case we remove the
                    // deferred state so that the state is reinitialised the next time this
                    // validation runs
                    val validationState: Any?
                    try {
                        validationState = deferredState.await()
                        FormManager.logger.trace {
                            "At '$path': Obtained validation '$validation' state: $validationState"
                        }
                    } catch (ex: Throwable) {
                        FormManager.logger.trace {
                            "At '$path': Failed to obtain validation '$validation' state ($ex)"
                        }
                        // [ex] cannot be a [CancellationException] because state updates should
                        // never be cancellable
                        state.setStatefulValidationDeferredState(null, statefulValidationIndex)
                        throw ex
                    }
                    validation.run { validationContext.validateFromState(validationState) }
                } else {
                    validation as Validation<Any?>
                    validation.run { validationContext.validate() }
                }
        } catch (ex: Throwable) {
            // The initialisation of the state of a stateful validation may be cancelled
            if (ex !is CancellationException) {
                FormManager.logger.error(ex) {
                    "At '$path': Failed to run validation '$validation'"
                }
                val failure = ValidationFailure(validation, ex)
                state.cacheIssues(validationIndex, arrayOf(failure))
                emit(failure)
            }
            return@flow
        }

        FormManager.logger.trace { "At '$path': Running validation '$validation'" }
        // If the validation throws, we still emit the issues up to the point it threw, plus a
        // [ValidationFailure] error
        val issuesToCache = mutableSetOf<ValidationIssue>()
        issuesFlow
            .catch { ex ->
                if (ex !is CancellationException) {
                    FormManager.logger.error(ex) {
                        "At '$path': Failed to run validation '$validation'"
                    }
                    val failure = ValidationFailure(validation, ex)
                    issuesToCache += failure
                    emit(failure)
                }
            }
            .onCompletion { ex ->
                // We mustn't cache issues during a cancellation as the cached issues must be
                // "complete"
                if (ex !is CancellationException) {
                    state.cacheIssues(validationIndex, issuesToCache.toTypedArray())
                }
            }
            .collect { issue ->
                issuesToCache += issue
                emit(issue)
            }
    }

    /** Initialises the state of a stateful validation. */
    private suspend fun initStatefulValidationState(
        path: AbsolutePath,
        state: StateImpl,
        validation: StatefulValidation<Any?, Any?>,
        statefulValidationIndex: Int,
        validationContext: ValidationContext,
    ): CompletableDeferred<Any?> {
        FormManager.logger.trace { "At '$path': Initializing validation '$validation' state" }
        val validationState = validation.run { validationContext.initState() }
        val deferredState = CompletableDeferred(validationState)
        state.setStatefulValidationDeferredState(deferredState, statefulValidationIndex)
        FormManager.logger.debug {
            "At '$path': Initialized validation '$validation' state: $validationState"
        }
        return deferredState
    }

    companion object {
        /** Updates the validation state, emitting events as needed. */
        suspend fun updateValidationState(
            formManager: FormManager,
            schema: Schema<*>,
            path: AbsolutePath,
            state: StateImpl,
            oldLocalDisplayStatus: DisplayStatus,
            oldDisplayStatus: DisplayStatus,
            descendantsDisplayingIssuesToUpdate:
                MutableMap<AbsolutePath, DescendantsDisplayingIssuesToUpdate>,
            emitValidationChange: Boolean = true,
        ) {
            if (emitValidationChange) {
                formManager.eventsBus.emit(
                    StateEvent.ValidationChange(
                        state.validationStatus,
                        state.getAllVisibleIssues(),
                        path,
                        schema,
                    )
                )
            }

            // Update display status if it changed
            val newDisplayStatus = state.displayStatus()
            if (oldDisplayStatus != newDisplayStatus) {
                formManager.eventsBus.emit(StateEvent.DisplayChange(newDisplayStatus, path, schema))
            }

            // Update count of "descendants displaying issues" on all ancestors, if necessary
            if (path != AbsolutePath.ROOT) {
                val newLocalDisplayStatus = state.localDisplayStatus()
                if (oldLocalDisplayStatus != newLocalDisplayStatus) {
                    updateDescendantsDisplayingIssues(
                        descendantsDisplayingIssuesToUpdate,
                        path.parent(),
                        if (oldLocalDisplayStatus == DisplayStatus.Error) -1
                        else if (newLocalDisplayStatus == DisplayStatus.Error) 1 else 0,
                        if (oldLocalDisplayStatus == DisplayStatus.Warning) -1
                        else if (newLocalDisplayStatus == DisplayStatus.Warning) 1 else 0,
                    )
                }
            }
        }
    }
}
