package tech.ostack.kform.internal.actions

import kotlin.plus
import kotlin.text.contains
import tech.ostack.kform.*
import tech.ostack.kform.internal.Action
import tech.ostack.kform.internal.INITIAL_VALUE
import tech.ostack.kform.internal.setOnParent

/**
 * Action that sets [toSet] as children of parent values matching [parentPath]. The children to set
 * are represented by [fragment]. [fragment] **cannot** be a recursive wildcard.
 *
 * To reset a value to its initial value, set [toSet] as [INITIAL_VALUE].
 */
internal class SetAction(
    formManager: FormManager,
    val parentPath: AbsolutePath?,
    parentSchema: ParentSchema<Any?>?,
    val fragment: AbsolutePathFragment?,
    val toSet: Any?,
) : WriteValueStateAction<Unit>(formManager) {
    override fun toString() = if (toSet === INITIAL_VALUE) "Reset($path)" else "Set($path, $toSet)"

    // When the parent schema supports concurrent sets, then we can simply lock the exact child
    // being set instead of the whole parent value
    override val accessedPaths =
        listOf(
            (if (parentSchema?.supportsConcurrentSets == true) path
            else parentPath ?: AbsolutePath.ROOT) + AbsolutePathFragment.RecursiveWildcard
        )

    val path
        get() = if (parentPath != null) parentPath + fragment!! else AbsolutePath.ROOT

    override fun overridesConflictingAction(action: Action<*>): Boolean =
        (action is SetAction &&
            fragment !is AbsolutePathFragment.CollectionEnd &&
            (path + AbsolutePathFragment.RecursiveWildcard).contains(action.path)) ||
            (action is RemoveAction &&
                (path + AbsolutePathFragment.RecursiveWildcard).contains(action.parentPath))

    override suspend fun runWriteValueState() = setOnParent(parentPath, fragment, toSet)
}
