package tech.ostack.kform.internal.actions

import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.FormManager
import tech.ostack.kform.LocatedValidationIssue
import tech.ostack.kform.Schema
import tech.ostack.kform.internal.*

/** Action that removes all provided external issues from the form manager. */
internal class RemoveDependingExternalIssuesAction(
    formManager: FormManager,
    private val externalIssues: Iterable<LocatedValidationIssue>,
) : ValueStateAction<Unit>(formManager) {
    override fun toString() = "RemoveDependingExternalIssuesAction(${externalIssues})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Write),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )
    override val accessedPaths = externalIssues.mapTo(HashSet()) { it.path }

    override suspend fun runValueState() =
        externalIssues
            .groupBy { it.path }
            .forEach { (path, issues) ->
                stateInfo(path).forEach { info ->
                    removeDependingExternalIssues(
                        info.state as StateImpl?,
                        info.schema,
                        info.path,
                        issues,
                    )
                }
            }

    private suspend fun removeDependingExternalIssues(
        state: StateImpl?,
        schema: Schema<*>,
        path: AbsolutePath,
        issues: Iterable<LocatedValidationIssue>,
    ) {
        state ?: return

        val oldLocalDisplayStatus = state.localDisplayStatus()
        val oldDisplayStatus = state.displayStatus()

        if (state.removeExternalIssues(issues)) {
            // Update validation state with the new external issues
            ValidateAction.updateValidationState(
                formManager,
                schema,
                path,
                state,
                oldLocalDisplayStatus,
                oldDisplayStatus,
                descendantsDisplayingIssuesToUpdate,
            )
        }
    }
}
