package tech.ostack.kform.internal.actions

import tech.ostack.kform.*
import tech.ostack.kform.internal.*

/**
 * Action that removes the cached issues of the validation of the value at [path] with index
 * [validationIndex] given the [state] of the value.
 */
internal class RemoveCachedIssuesAction(
    formManager: FormManager,
    private val path: AbsolutePath,
    private val schema: Schema<*>,
    private val state: StateImpl,
    private val validation: Validation<*>,
    private val validationIndex: Int,
) : FormManagerAction<Unit>(formManager) {
    override fun toString(): String = "RemoveCachedIssues($validation@$path)"

    override val accesses =
        listOf(
            AccessValidationState(ActionAccessType.Write),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )
    override val accessedPaths = listOf(path)

    override suspend fun run() =
        removeCachedIssues(formManager, path, schema, state, validationIndex)

    companion object {
        /**
         * Function that removes the cached issues of validation with index [validationIndex] (or of
         * all validations when [validationIndex] is `null`) at [path].
         *
         * When appropriate, emits events about a change of validation/display status and emits an
         * action to update the number of descendants displaying issues on the parent value.
         */
        suspend fun removeCachedIssues(
            formManager: FormManager,
            path: AbsolutePath,
            schema: Schema<*>,
            state: StateImpl,
            validationIndex: Int? = null,
        ) {
            val wasValidated =
                state.validationStatus == ValidationStatus.Validated ||
                    state.validationStatus == ValidationStatus.ValidatedExceptionally
            val oldLocalDisplayStatus = state.localDisplayStatus()
            val oldDisplayStatus = state.displayStatus()

            state.removeCachedIssues(validationIndex)

            // Update validation state as needed
            if (wasValidated) {
                state.validationStatus = ValidationStatus.Unvalidated
            }

            val descendantsDisplayingIssuesToUpdate =
                mutableMapOf<AbsolutePath, DescendantsDisplayingIssuesToUpdate>()
            ValidateAction.updateValidationState(
                formManager,
                schema,
                path,
                state,
                oldLocalDisplayStatus,
                oldDisplayStatus,
                descendantsDisplayingIssuesToUpdate,
                emitValidationChange = wasValidated,
            )

            if (descendantsDisplayingIssuesToUpdate.isNotEmpty()) {
                formManager.scheduleAction(
                    UpdateDescendantsDisplayingIssuesAction(
                        formManager,
                        descendantsDisplayingIssuesToUpdate.values,
                    )
                )
            }
        }
    }
}
