package tech.ostack.kform.internal.actions

import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.FormManager
import tech.ostack.kform.Schema
import tech.ostack.kform.Validation
import tech.ostack.kform.internal.*

/** Information about a validation to be invalidated. */
internal class ValidationToInvalidate(
    val path: AbsolutePath,
    val validation: Validation<*>,
    val validationIndex: Int,
) {
    override fun equals(other: Any?) =
        when {
            this === other -> true
            other !is ValidationToInvalidate -> false
            else -> path == other.path && validationIndex == other.validationIndex
        }

    override fun hashCode(): Int {
        var result = path.hashCode()
        result = 31 * result + validationIndex
        return result
    }
}

/**
 * Action that invalidates all provided validations by removing their cached issues and marking the
 * values as unvalidated when necessary.
 */
internal class InvalidateValidationsAction(
    formManager: FormManager,
    private val toInvalidate: Set<ValidationToInvalidate>,
) : ValueStateAction<Unit>(formManager) {
    override fun toString(): String =
        "InvalidateValidations(${toInvalidate.joinToString { "${it.validation}@${it.path}" }})"

    override val accesses =
        listOf(
            AccessValueStateTree(ActionAccessType.Read),
            AccessValidationState(ActionAccessType.Write),
            AccessIsTouched(ActionAccessType.Read),
            AccessDescendantsDisplayingIssues(ActionAccessType.Read),
        )
    override val accessedPaths = toInvalidate.map { it.path }.toHashSet()

    override fun overridesConflictingAction(action: Action<*>): Boolean =
        action is InvalidateValidationsAction && toInvalidate.containsAll(action.toInvalidate)

    override suspend fun runValueState() =
        toInvalidate.forEach { validationInfo ->
            stateInfo(validationInfo.path).forEach { info ->
                invalidateValidation(
                    info.path,
                    info.schema,
                    info.state as StateImpl?,
                    validationInfo.validationIndex,
                )
            }
        }

    private suspend fun invalidateValidation(
        path: AbsolutePath,
        schema: Schema<*>,
        state: StateImpl?,
        validationIndex: Int,
    ) {
        state ?: return
        RemoveCachedIssuesAction.removeCachedIssues(
            formManager,
            path,
            schema,
            state,
            validationIndex,
        )
    }
}
