package tech.ostack.kform.internal.actions

import kotlinx.coroutines.CancellationException
import kotlinx.coroutines.CompletableDeferred
import tech.ostack.kform.*
import tech.ostack.kform.internal.*
import tech.ostack.kform.schemas.ComputedSchema

/** Action that computes and sets the computed values of all values at paths [toCompute]. */
internal class ComputeValuesAction(
    formManager: FormManager,
    val toCompute: Iterable<AbsolutePath>,
) : WriteValueStateAction<Unit>(formManager) {
    override fun toString() = "ComputeValues(${toCompute.joinToString()})"

    override val accesses =
        mutableListOf(
            AccessValueStateTree(ActionAccessType.Write),
            AccessStatefulComputedValueDeferredState(ActionAccessType.Read),
        )

    override val accessedPaths: Set<AbsolutePath>

    init {
        // Iterate over all computed values that need to run and determine the accessed paths and
        // whether access to external context is required
        var requiresAccessToExternalContext = false
        val paths = mutableSetOf<AbsolutePath>()

        for (path in toCompute) {
            for ((schema, _, queriedPath) in schemaInfo(path)) {
                if (schema !is ComputedSchema) {
                    throw AtPathException(path, "Schema is not computed.")
                }

                // Parent required for setting the value
                paths += queriedPath.parent()
                paths += queriedPath + AbsolutePathFragment.RecursiveWildcard
                for (dependency in schema.computedValue.dependencies.values) {
                    paths +=
                        queriedPath.resolve(dependency.path) +
                            AbsolutePathFragment.RecursiveWildcard
                }
                if (schema.computedValue.externalContextDependencies.isNotEmpty()) {
                    requiresAccessToExternalContext = true
                }
            }
        }

        if (requiresAccessToExternalContext) {
            accesses += AccessExternalContext(ActionAccessType.Read)
        }
        accessedPaths = paths
    }

    override suspend fun runWriteValueState() =
        toCompute.forEach {
            valueStateInfo(it).collect { (_, state, schema, path, schemaPath) ->
                compute(state as StateImpl?, schema as ComputedSchema<*>, path, schemaPath)
            }
        }

    /** Computes all computed values matching the action's path. */
    private suspend fun compute(
        state: StateImpl?,
        schema: ComputedSchema<*>,
        path: AbsolutePath,
        schemaPath: AbsolutePath,
    ) {
        state ?: return

        try {
            val result = runComputedValue(schema, path, schemaPath, state, schema.computedValue)
            setOnParent(path.parent(), path.lastFragment, result)
        } catch (_: Throwable) {}
    }

    /** Computes a single computed value. This function may throw a `CancellationException`. */
    private suspend fun runComputedValue(
        schema: ComputedSchema<*>,
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        state: StateImpl,
        computedValue: ComputedValue<*>,
    ): Any? {
        val computedValueContext =
            ComputedValueContext(
                schema,
                path,
                schemaPath,
                dependenciesInfo(path, computedValue.dependencies),
                computedValue.externalContextDependencies.associateWith {
                    formManager.externalContexts[it]
                },
            )

        var result: Any?
        try {
            if (computedValue is StatefulComputedValue<*, *>) {
                @Suppress("UNCHECKED_CAST")
                computedValue as StatefulComputedValue<Any?, Any?>

                var deferredState = state.statefulComputedValueDeferredState
                // Initialise computed value state if it has not yet been initialised; this may
                // fail due to an error in the `initState` function or due to cancellation
                if (deferredState == null) {
                    deferredState =
                        initStatefulComputedValueState(
                            path,
                            state,
                            computedValue,
                            computedValueContext,
                        )
                }
                // Wait for the computed value state in case ongoing state updates exist; this may
                // fail due to an error during the state update, in which case we remove the
                // deferred state so that the state is reinitialised the next time this computed
                // value runs
                val computedValueState: Any?
                try {
                    computedValueState = deferredState.await()
                    FormManager.logger.trace {
                        "At '$path': Obtained computed value '$computedValue' state: $computedValueState"
                    }
                } catch (ex: Throwable) {
                    FormManager.logger.trace {
                        "At '$path': Failed to obtain computed value '$computedValue' state ($ex)"
                    }
                    // [ex] cannot be a [CancellationException] because state updates should
                    // never be cancellable
                    state.statefulComputedValueDeferredState = null
                    throw ex
                }

                FormManager.logger.trace { "At '$path': Computing value '$computedValue'" }
                result =
                    computedValue.run { computedValueContext.computeFromState(computedValueState) }
            } else {
                FormManager.logger.trace { "At '$path': Computing value '$computedValue'" }
                result = computedValue.run { computedValueContext.compute() }
            }
        } catch (ex: Throwable) {
            // The initialisation of the state of a stateful validation may be cancelled
            if (ex !is CancellationException) {
                FormManager.logger.error(ex) {
                    "At '$path': Failed to compute value '$computedValue'"
                }
            }
            throw ex
        }

        return result
    }

    /** Initialises the state of a computed value. */
    private suspend fun initStatefulComputedValueState(
        path: AbsolutePath,
        state: StateImpl,
        computedValue: StatefulComputedValue<Any?, Any?>,
        computedValueContext: ComputedValueContext,
    ): CompletableDeferred<Any?> {
        FormManager.logger.trace {
            "At '$path': Initializing computed value '$computedValue' state"
        }
        val computedValueState = computedValue.run { computedValueContext.initState() }
        val deferredState = CompletableDeferred(computedValueState)
        state.statefulComputedValueDeferredState = deferredState
        FormManager.logger.debug {
            "At '$path': Initialized computed value '$computedValue' state: $computedValueState"
        }
        return deferredState
    }
}
