package tech.ostack.kform.internal

import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.cancel
import tech.ostack.kform.*

/**
 * Implementation of the state associated with a value in the form manager.
 *
 * The value associated with this state is being validated by a provided number of validations
 * `nValidations` and a provided number of stateful validations `nStatefulValidations`.
 */
internal open class StateImpl(nValidations: Int, nStatefulValidations: Int) : State {
    /**
     * Cached issues of each validation. Elements are `null` when the issues of a certain validation
     * are unknown.
     */
    private val cachedIssues: Array<Array<ValidationIssue>?>? =
        if (nValidations == 0) null else arrayOfNulls(nValidations)

    /**
     * Deferred state of each stateful validation. Elements are `null` when the state hasn't been
     * initialised.
     */
    private val statefulValidationsDeferredState: Array<CompletableDeferred<Any?>?>? =
        if (nStatefulValidations == 0) null else arrayOfNulls(nStatefulValidations)

    /** Validation status of the value. */
    var validationStatus: ValidationStatus =
        if (nValidations > 0) ValidationStatus.Unvalidated else ValidationStatus.Validated

    /** External issues added to the value. */
    var externalIssues: Array<LocatedValidationIssue>? = null

    /**
     * Deferred state of the stateful computed value. `null` when the state hasn't been initialised.
     */
    var statefulComputedValueDeferredState: CompletableDeferred<Any?>? = null

    /** Whether the value associated with this state instance is dirty. */
    var isDirty = false

    /** Whether the value associated with this state instance has been touched. */
    var isTouched = false

    /**
     * Returns the cached issues of the validation with index [validationIndex], or `null` if no
     * issues have been cached for that validation.
     */
    fun getCachedIssues(validationIndex: Int): Array<ValidationIssue>? =
        cachedIssues!![validationIndex]

    /** Returns a list with all cached issues. */
    private fun getAllCachedIssues(): Sequence<ValidationIssue> = sequence {
        cachedIssues ?: return@sequence
        for (issues in cachedIssues) {
            issues ?: continue
            for (issue in issues) {
                yield(issue)
            }
        }
    }

    /** Caches the [issues] of the validation with index [validationIndex]. */
    fun cacheIssues(validationIndex: Int, issues: Array<ValidationIssue>) {
        cachedIssues!![validationIndex] = issues
    }

    /**
     * Removes the cached issues of the validation with index [validationIndex] or all cached issues
     * when [validationIndex] is `-1`.
     */
    fun removeCachedIssues(validationIndex: Int? = null) {
        if (validationIndex == null) {
            cachedIssues?.fill(null)
        } else {
            cachedIssues!![validationIndex] = null
        }
    }

    /**
     * Returns a list of all (distinct) issues that should be visible to the user. I.e. all chached
     * issues (if the validation status is validated) + all external issues.
     */
    fun getAllVisibleIssues(): List<ValidationIssue> {
        val nonLocatedExternalIssues =
            externalIssues?.map { ValidationIssue(it.code, it.severity, it.data) } ?: emptyList()
        val visibleIssues =
            if (
                validationStatus == ValidationStatus.Validated ||
                    validationStatus == ValidationStatus.ValidatedExceptionally
            )
                getAllCachedIssues().toList() + nonLocatedExternalIssues
            else nonLocatedExternalIssues
        return visibleIssues.distinct()
    }

    /** Adds external issues to the value. */
    fun addExternalIssues(issues: Iterable<LocatedValidationIssue>) {
        externalIssues =
            if (externalIssues != null)
                externalIssues!!.toMutableSet().let {
                    it.addAll(issues)
                    it.toTypedArray()
                }
            else issues.toSet().toTypedArray()
    }

    /**
     * Removed all external issues from the value with the provided code, or all issues if no code
     * is provided. Returns a list containing the removed issues.
     */
    fun removeExternalIssues(code: String? = null): List<LocatedValidationIssue> {
        if (externalIssues == null) {
            return emptyList()
        }
        if (code == null) {
            val toRemove = externalIssues!!.toList()
            externalIssues = null
            return toRemove
        }
        val (toRemove, toKeep) = externalIssues!!.partition { it.code == code }
        externalIssues = if (toKeep.isEmpty()) null else toKeep.toTypedArray()
        return toRemove
    }

    /**
     * Removes the provided external issues from the list of external issues. Returns whether at
     * least one issue was removed.
     */
    fun removeExternalIssues(issues: Iterable<LocatedValidationIssue>): Boolean {
        if (externalIssues == null) {
            return false
        }
        val toRemove = issues.toSet()
        val toKeep = externalIssues!!.filter { it !in toRemove }
        if (externalIssues!!.size != toKeep.size) {
            externalIssues = toKeep.toTypedArray()
            return true
        }
        return false
    }

    /**
     * Returns the deferred validation state of the stateful validation with index
     * [statefulValidationIndex].
     */
    fun getStatefulValidationDeferredState(statefulValidationIndex: Int) =
        statefulValidationsDeferredState!![statefulValidationIndex]

    /**
     * Sets the deferred validation state of the stateful validation with index
     * [statefulValidationIndex].
     */
    fun setStatefulValidationDeferredState(
        statefulValidationDeferredState: CompletableDeferred<Any?>?,
        statefulValidationIndex: Int,
    ) {
        statefulValidationsDeferredState!![statefulValidationIndex] =
            statefulValidationDeferredState
    }

    /** Destroy the state by cancelling all deferred computation states. */
    fun destroy() {
        if (statefulValidationsDeferredState != null) {
            for (deferredState in statefulValidationsDeferredState) {
                deferredState?.cancel()
            }
        }
        statefulComputedValueDeferredState?.cancel()
    }

    /**
     * Dominant issue severity of the all issues (cached + external): error severity if at least one
     * issue is an error, or warning severity when all issues are warnings.
     *
     * Returns `null` when the value has no external issues and either has not been fully validated
     * or is valid.
     */
    fun dominantIssueSeverity(): ValidationIssueSeverity? {
        if (validationStatus == ValidationStatus.ValidatedExceptionally) {
            return ValidationIssueSeverity.Error
        }
        var severity: ValidationIssueSeverity? = null
        if (externalIssues != null) {
            for (externalIssue in externalIssues!!) {
                if (externalIssue.severity == ValidationIssueSeverity.Error) {
                    return ValidationIssueSeverity.Error
                } else {
                    severity = ValidationIssueSeverity.Warning
                }
            }
        }
        if (validationStatus == ValidationStatus.Validated) {
            for (cachedIssue in getAllCachedIssues()) {
                if (cachedIssue.severity == ValidationIssueSeverity.Error) {
                    return ValidationIssueSeverity.Error
                } else {
                    severity = ValidationIssueSeverity.Warning
                }
            }
        }
        return severity
    }

    /**
     * Local display status of the value, ignoring display status of descendants. This function
     * reads validation status, external issues, and touched status.
     */
    fun localDisplayStatus(): DisplayStatus {
        if (!isTouched) {
            return DisplayStatus.Valid
        }
        return when (dominantIssueSeverity()) {
            ValidationIssueSeverity.Error -> DisplayStatus.Error
            ValidationIssueSeverity.Warning -> DisplayStatus.Warning
            null -> DisplayStatus.Valid
        }
    }

    /**
     * Current display status of the value. This function reads validation status, touched status,
     * and (when a parent state) the number of descendants displaying issues.
     */
    open fun displayStatus(): DisplayStatus = localDisplayStatus()
}

/** Implementation of the state associated with a parent value in the form manager. */
internal open class ParentStateImpl(
    parentState: ParentState,
    nValidations: Int,
    nStatefulValidations: Int,
) : StateImpl(nValidations, nStatefulValidations), ParentState by parentState {
    /** Number of descendants displaying errors. */
    var descendantsDisplayingErrors: Int = 0

    /** Number of descendants displaying warnings. */
    var descendantsDisplayingWarnings: Int = 0

    override fun displayStatus(): DisplayStatus {
        if (!isTouched) {
            return DisplayStatus.Valid
        }
        if (descendantsDisplayingErrors > 0) {
            return DisplayStatus.Error
        }
        val dominantSeverity = dominantIssueSeverity()
        if (dominantSeverity != null) {
            return when (dominantSeverity) {
                ValidationIssueSeverity.Error -> DisplayStatus.Error
                ValidationIssueSeverity.Warning -> DisplayStatus.Warning
            }
        }
        if (descendantsDisplayingWarnings > 0) {
            return DisplayStatus.Warning
        }
        return DisplayStatus.Valid
    }
}

/** Implementation of the state associated with a collection in the form manager. */
internal class CollectionStateImpl(
    private val collectionState: CollectionState,
    nValidations: Int,
    nStatefulValidations: Int,
) :
    ParentStateImpl(collectionState, nValidations, nStatefulValidations),
    CollectionState by collectionState {
    override fun childrenStates(
        path: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<StateInfo<*>> = collectionState.childrenStates(path, fragment)

    override fun setState(fragment: AbsolutePathFragment.Id, state: State?) =
        collectionState.setState(fragment, state)
}
