@file:JvmName("PathMultimaps")

package tech.ostack.kform.collections

import kotlin.jvm.JvmName
import tech.ostack.kform.AbsolutePath
import tech.ostack.kform.Path
import tech.ostack.kform.PathOrString
import tech.ostack.kform.toAbsolutePath

/** Identifier of an entry in a path multimap. */
public typealias PathMultimapEntryId = Int

/** An entry of the path multimap with values of type [T]. */
public interface PathMultimapEntry<out T> {
    /** Path of the multimap entry. */
    public val path: AbsolutePath

    public operator fun component1(): AbsolutePath = path

    /** Value of the multimap entry. */
    public val value: T

    public operator fun component2(): T = value

    /** Unique identified of the entry in the multimap. */
    public val id: PathMultimapEntryId

    public operator fun component3(): PathMultimapEntryId = id
}

/**
 * Multimap implementation mapping paths to values of type [T].
 *
 * It is a multimap in the sense that multiple values may be associated with the same path. However,
 * each entry in the multimap is represented by an id [PathMultimapEntryId] and the multimap allows
 * operations on said ids.
 *
 * Operations on the multimap that receive paths follow the path semantics of
 * [matching][AbsolutePath.matches] (for [contains], [get], and [entries]). This allows us to, for
 * example, get all values matching a certain path that contains wildcards; or to have entries
 * containing paths with wildcards in the multimap whose value will be returned when getting a path
 * without wildcards that matches it.
 */
public interface PathMultimap<out T> {
    /** Number of entries in the multimap. */
    public val size: Int

    /** Whether the multimap is empty. */
    public fun isEmpty(): Boolean = size == 0

    /** Sequence of all entries in the multimap (may have repeated entries). */
    public val entries: Sequence<PathMultimapEntry<T>>
        get() = entries(AbsolutePath.MATCH_ALL)

    /** Sequence of values in the multimap (may have repeated values). */
    public val values: Sequence<T>
        get() = entries.map { entry -> entry.value }

    /**
     * Returns whether there exists at least one entry with a path matching [path] (following the
     * semantics of [path matching][AbsolutePath.matches]).
     */
    public fun containsPath(path: Path): Boolean

    /**
     * Returns whether there exists at least one entry with a path matching [path] (following the
     * semantics of [path matching][AbsolutePath.matches]).
     */
    public fun containsPath(path: String): Boolean = containsPath(AbsolutePath(path))

    /**
     * Returns whether there exists at least one entry with a path matching [path] (following the
     * semantics of [path matching][AbsolutePath.matches]). Equivalent to `containsPath(path)`.
     */
    public operator fun contains(path: Path): Boolean = containsPath(path)

    /**
     * Returns whether there exists at least one entry with a path matching [path] (following the
     * semantics of [path matching][AbsolutePath.matches]). Equivalent to `containsPath(path)`.
     */
    public operator fun contains(path: String): Boolean = containsPath(path)

    /** Returns whether there exists an entry identified by [entryId] in the multimap. */
    public fun containsEntry(entryId: PathMultimapEntryId): Boolean

    /**
     * Returns whether there exists an entry identified by [entryId] in the multimap. Equivalent to
     * `containsEntry(entryId)`.
     */
    public operator fun contains(entryId: PathMultimapEntryId): Boolean = containsEntry(entryId)

    /** Returns whether there exists at least one value equal to [value] in the multimap. */
    public fun containsValue(value: @UnsafeVariance T): Boolean

    /**
     * Returns whether there exists at least one value equal to [value] in the multimap. Equivalent
     * to `containsValue(value)`.
     */
    public operator fun contains(value: @UnsafeVariance T): Boolean = containsValue(value)

    /**
     * Returns a sequence over all values with a path matching [path] (following the semantics of
     * [path matching][AbsolutePath.matches]).
     */
    public operator fun get(path: Path): Sequence<T>

    /**
     * Returns a sequence over all values with a path matching [path] (following the semantics of
     * [path matching][AbsolutePath.matches]).
     */
    public operator fun get(path: String): Sequence<T> = get(AbsolutePath(path))

    /** Returns the entry identified by [entryId] or `null` when no such entry exists. */
    public fun getEntry(entryId: PathMultimapEntryId): PathMultimapEntry<T>?

    /**
     * Returns the entry identified by [entryId] or `null` when no such entry exists. Equivalent to
     * `getEntry(entryId)`.
     */
    public operator fun get(entryId: PathMultimapEntryId): PathMultimapEntry<T>? = getEntry(entryId)

    /**
     * Returns one value matching [path] or `null` when no such value exists. Equivalent to
     * `get(path).firstOrNull()`.
     */
    public fun getOne(path: Path): T? = get(path).firstOrNull()

    /**
     * Returns one value matching [path] or `null` when no such value exists. Equivalent to
     * `get(path).firstOrNull()`.
     */
    public fun getOne(path: String): T? = getOne(AbsolutePath(path))

    /**
     * Returns a sequence over all entries with a path matching [path] (following the semantics of
     * [path matching][AbsolutePath.matches]).
     */
    public fun entries(path: Path = AbsolutePath.MATCH_ALL): Sequence<PathMultimapEntry<T>>

    /**
     * Returns a sequence over all entries with a path matching [path] (following the semantics of
     * [path matching][AbsolutePath.matches]).
     */
    public fun entries(path: String): Sequence<PathMultimapEntry<T>> = entries(AbsolutePath(path))
}

/**
 * Mutable multimap implementation mapping paths to values of type [T]. Extends [PathMultimap] with
 * operations to mutate the multimap.
 */
public interface MutablePathMultimap<T> : PathMultimap<T> {
    /**
     * Inserts a new entry in the multimap with the given [path] and [value] and returns an
     * identifier that identifies the inserted entry in the multimap.
     *
     * The returned identifier can be used to remove the inserted entry from the multimap.
     */
    public fun put(path: Path, value: T): PathMultimapEntryId

    /**
     * Inserts a new entry in the multimap with the given [path] and [value] and returns an
     * identifier that identifies the inserted entry in the multimap.
     *
     * The returned identifier can be used to remove the inserted entry from the multimap.
     */
    public fun put(path: String, value: T): PathMultimapEntryId = put(AbsolutePath(path), value)

    /**
     * Inserts all entries of [pathMultimap] into this multimap and returns a list with the
     * identifiers of each inserted entry.
     */
    public fun putAll(pathMultimap: PathMultimap<T>): List<PathMultimapEntryId> {
        val removeFuns = mutableListOf<PathMultimapEntryId>()
        for ((path, value) in pathMultimap.entries) {
            removeFuns += put(path, value)
        }
        return removeFuns
    }

    /**
     * Removes all entries with a path contained by [path] (following the semantics of
     * [path containment][AbsolutePath.contains]) and returns a list of said entries.
     */
    public fun remove(path: Path): List<PathMultimapEntry<T>>

    /**
     * Removes all entries with a path contained by [path] (following the semantics of
     * [path containment][AbsolutePath.contains]) and returns a list of said entries.
     */
    public fun remove(path: String): List<PathMultimapEntry<T>> = remove(AbsolutePath(path))

    /**
     * Removes the entry identified by [entryId] and returns it or `null` when no such entry exists.
     */
    public fun removeEntry(entryId: PathMultimapEntryId): PathMultimapEntry<T>?

    /** Removes all entries from the path multimap. */
    public fun clear()
}

/**
 * Inserts a new entry in the multimap with the given [path] and [value] and returns an identifier
 * that identifies the inserted entry in the multimap.
 *
 * The returned identifier can be used to remove the inserted entry from the multimap.
 */
public operator fun <T> MutablePathMultimap<T>.set(path: Path, value: T): PathMultimapEntryId =
    put(path, value)

/**
 * Inserts a new entry in the multimap with the given [path] and [value] and returns an identifier
 * that identifies the inserted entry in the multimap.
 *
 * The returned identifier can be used to remove the inserted entry from the multimap.
 */
public operator fun <T> MutablePathMultimap<T>.set(path: String, value: T): PathMultimapEntryId =
    put(AbsolutePath(path), value)

/**
 * Returns a new read-only path multimap given a list of [pairs] where the first value is the path
 * and the second is the value.
 */
public fun <T> pathMultimapOf(vararg pairs: Pair<PathOrString, T>): PathMultimap<T> =
    mutablePathMultimapOf(*pairs)

/**
 * Returns a new mutable path multimap given a list of [pairs] where the first value is the path and
 * the second is the value.
 *
 * For removal purposes and for operations on ids, the first entry has id `0L`, the second entry id
 * `1L`, etc.
 */
public fun <T> mutablePathMultimapOf(vararg pairs: Pair<PathOrString, T>): MutablePathMultimap<T> {
    val trie: PathTrie<T> = if (pairs.isEmpty()) PathTrie() else PathTrie(pairs.size)
    for (pair in pairs) {
        trie[pair.first.toAbsolutePath()] = pair.second
    }
    return trie
}

/** Transforms the multimap into a mapping of paths to lists of values associated with them. */
public fun <T> PathMultimap<T>.toMap(): MutableMap<AbsolutePath, MutableList<T>> {
    val map: MutableMap<AbsolutePath, MutableList<T>> = LinkedHashMap(size)
    for (entry in entries) {
        map.getOrPut(entry.path) { mutableListOf() } += entry.value
    }
    return map
}
