@file:JvmName("ValidationIssues")

package tech.ostack.kform

import kotlin.js.JsName
import kotlin.jvm.JvmName
import kotlin.jvm.JvmOverloads
import kotlin.jvm.JvmStatic
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.firstOrNull
import kotlinx.serialization.SerialName
import kotlinx.serialization.Serializable

/**
 * Data associated with a validation issue. Useful, for example, for displaying (possibly located)
 * messages associated with the issue.
 */
// TODO: Possibly allow entry values other than string (currently not supported to simplify
//  serialisation); related: https://github.com/Kotlin/kotlinx.serialization/issues/49
//  Could use `@Polymorphic Any?` and provide a serialisation module to be used by apps.
public typealias ValidationIssueData = Map<String, String?> // Map<String, @Polymorphic Any?>?

/** Severity of a validation issue. */
public enum class ValidationIssueSeverity {
    Error,
    Warning,
}

/**
 * Validation issue emitted by a validation indicating a (potential) problem with the form data.
 *
 * A validation issue can be either a [ValidationError], [ValidationWarning], or, exceptionally, a
 * [ValidationFailure].
 */
@JsName("ValidationIssueKt")
@Serializable
public sealed class ValidationIssue {
    /** Code representing the issue. */
    public abstract val code: String

    /**
     * Additional issue data. This additional data can be used, for example, to help write a
     * user-friendly message associated with the issue.
     */
    public abstract val data: ValidationIssueData

    /** Severity of the validation issue. */
    public val severity: ValidationIssueSeverity
        get() =
            when (this) {
                is ValidationError,
                is ValidationFailure -> ValidationIssueSeverity.Error
                is ValidationWarning -> ValidationIssueSeverity.Warning
            }

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is ValidationIssue -> false
            else -> code == other.code && severity == other.severity && data == other.data
        }

    override fun hashCode(): Int {
        var result = code.hashCode()
        result = 31 * result + severity.hashCode()
        result = 31 * result + data.hashCode()
        return result
    }

    /**
     * Whether this validation issue "contains" [issue].
     *
     * This validation issue is said to contain [issue] if both have the same code, the same
     * severity, and if the [issue]'s data is either `null` or every one of its entries exists in
     * this issue's data.
     */
    public fun contains(issue: ValidationIssue): Boolean =
        code == issue.code &&
            severity == issue.severity &&
            issue.data.all { (key, value) -> key in data && data[key] == value }

    public companion object {
        /**
         * Creates a [ValidationIssue] with the provided [code], [severity], and [data].
         *
         * When [severity] is [ValidationIssueSeverity.Error], a [ValidationError] is created;
         * otherwise, a [ValidationWarning] is created.
         */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun invoke(
            code: String,
            severity: ValidationIssueSeverity,
            data: ValidationIssueData = emptyMap(),
        ): ValidationIssue =
            when (severity) {
                ValidationIssueSeverity.Error -> ValidationError(code, data)
                ValidationIssueSeverity.Warning -> ValidationWarning(code, data)
            }
    }
}

/**
 * Validation error emitted by a validation indicating a problem with the form data.
 *
 * A form is considered invalid if it contains at least one validation error.
 */
@JsName("ValidationErrorKt")
@Serializable
@SerialName("error")
public class ValidationError
@JvmOverloads
constructor(override val code: String, override val data: ValidationIssueData = emptyMap()) :
    ValidationIssue() {
    override fun toString(): String = buildString {
        append("ValidationError(code=$code")
        if (data.isNotEmpty()) {
            append(", data=$data")
        }
        append(")")
    }
}

/**
 * Validation warning emitted by a validation indicating a potential problem with the form data.
 *
 * A form is **not** considered invalid if it contains only warnings.
 */
@JsName("ValidationWarningKt")
@Serializable
@SerialName("warning")
public class ValidationWarning
@JvmOverloads
constructor(override val code: String, override val data: ValidationIssueData = emptyMap()) :
    ValidationIssue() {
    override fun toString(): String = buildString {
        append("ValidationWarning(code=$code")
        if (data.isNotEmpty()) {
            append(", data=$data")
        }
        append(")")
    }
}

/**
 * Validation issue built by the [manager][FormManager] when a validation fails to run (throws an
 * exception while being executed). The code of this validation error is [CODE].
 *
 * The validation error [data] contains entries with keys `"validation"`, `"exception"`, and
 * `"stackTrace"` containing information on the name of the validation that caused the exception as
 * well as information about the exception itself. These data properties are also accessible via
 * properties of the class with the same name.
 */
@JsName("ValidationFailureKt")
@Serializable
@SerialName("failure")
public class ValidationFailure
internal constructor(
    override val code: String,
    override val data: ValidationIssueData = emptyMap(),
) : ValidationIssue() {
    internal constructor(
        validation: Validation<*>,
        throwable: Throwable,
    ) : this(
        CODE,
        mapOf(
            "validation" to validation.toString(),
            "exception" to
                listOfNotNull(throwable::class.simpleName, throwable.message)
                    .joinToString(": ")
                    .ifEmpty { null },
            "stackTrace" to throwable.stackTraceToString(),
        ),
    )

    /** Name of the validation where the exception occurred. */
    public val validation: String
        get() = data["validation"]!!

    /** Exception that occurred (name and message). */
    public val exception: String?
        get() = data["exception"]

    /** Exception stack trace. */
    public val stackTrace: String
        get() = data["stackTrace"]!!

    override fun toString(): String = buildString {
        append("ValidationFailure(code=$code, validation=$validation")
        if (exception != null) {
            append(", exception=$exception")
        }
        append(")")
    }

    public companion object {
        /** Validation error code representing that the validation failed to run. */
        public const val CODE: String = "validationFailed"
    }
}

/**
 * Validation issue emitted by a [form validator][FormValidator] or [manager][FormManager]
 * containing location information.
 *
 * A located validation issue can be either a [LocatedValidationError] or a
 * [LocatedValidationWarning].
 */
@JsName("LocatedValidationIssueKt")
@Serializable
public sealed class LocatedValidationIssue {
    /** Path of the value containing the issue. */
    public abstract val path: AbsolutePath

    /** Code representing the issue. */
    public abstract val code: String

    /** Paths of values that the validation that produced this issue depends on. */
    public abstract val dependencies: Set<AbsolutePath>

    /**
     * Whether the validation that produced this issue depends on the descendants of the validated
     * value.
     */
    public abstract val dependsOnDescendants: Boolean

    /** External contexts that the validation that produced this issue depends on. */
    public abstract val externalContextDependencies: Set<String>

    /**
     * Additional issue data. This additional data can be used, for example, to help write a
     * user-friendly message associated with the issue.
     */
    public abstract val data: ValidationIssueData

    /** Severity of the located validation issue. */
    public val severity: ValidationIssueSeverity
        get() =
            when (this) {
                is LocatedValidationError,
                is LocatedValidationFailure -> ValidationIssueSeverity.Error
                is LocatedValidationWarning -> ValidationIssueSeverity.Warning
            }

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is LocatedValidationIssue -> false
            else ->
                path == other.path &&
                    code == other.code &&
                    severity == other.severity &&
                    dependencies == other.dependencies &&
                    dependsOnDescendants == other.dependsOnDescendants &&
                    data == other.data
        }

    override fun hashCode(): Int {
        var result = path.hashCode()
        result = 31 * result + code.hashCode()
        result = 31 * result + severity.hashCode()
        result = 31 * result + dependencies.hashCode()
        result = 31 * result + dependsOnDescendants.hashCode()
        result = 31 * result + data.hashCode()
        return result
    }

    /**
     * Whether this located validation issue "contains" [issue].
     *
     * This validation issue is said to contain [issue] if both have the same path, the same code,
     * the same severity, if the set of this issue's dependencies contains the set of [issue]'s
     * dependencies, if the [issue]'s [dependsOnDescendants] is either `false` or equal to this
     * issue's [dependsOnDescendants], and if the [issue]'s data is either `null` or every one of
     * its entries exists in this issue's data.
     */
    public fun contains(issue: LocatedValidationIssue): Boolean =
        path == issue.path &&
            code == issue.code &&
            severity == issue.severity &&
            (dependsOnDescendants || dependsOnDescendants == issue.dependsOnDescendants) &&
            dependencies.containsAll(issue.dependencies) &&
            issue.data.all { (key, value) -> key in data && data[key] == value }

    public companion object {
        /**
         * Creates a [LocatedValidationIssue] with the provided [path], [code], [severity],
         * [dependencies], whether it [dependsOnDescendants], [externalContextDependencies], and
         * [data].
         *
         * When [severity] is [ValidationIssueSeverity.Error], a [LocatedValidationError] is
         * created; otherwise, a [LocatedValidationWarning] is created.
         */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun invoke(
            path: Path,
            code: String,
            severity: ValidationIssueSeverity,
            dependencies: Iterable<Path> = emptySet(),
            dependsOnDescendants: Boolean = false,
            externalContextDependencies: Iterable<String> = emptySet(),
            data: ValidationIssueData = emptyMap(),
        ): LocatedValidationIssue =
            when (severity) {
                ValidationIssueSeverity.Error ->
                    LocatedValidationError(
                        path,
                        code,
                        dependencies,
                        dependsOnDescendants,
                        externalContextDependencies,
                        data,
                    )
                ValidationIssueSeverity.Warning ->
                    LocatedValidationWarning(
                        path,
                        code,
                        dependencies,
                        dependsOnDescendants,
                        externalContextDependencies,
                        data,
                    )
            }

        /**
         * Creates a [LocatedValidationIssue] with the provided [path], [code], [severity],
         * [dependencies], whether it [dependsOnDescendants], [externalContextDependencies], and
         * [data].
         *
         * When [severity] is [ValidationIssueSeverity.Error], a [LocatedValidationError] is
         * created; otherwise, a [LocatedValidationWarning] is created.
         */
        @JvmOverloads
        @JvmStatic
        @JvmName("create")
        public operator fun invoke(
            path: String,
            code: String,
            severity: ValidationIssueSeverity,
            dependencies: Iterable<String> = emptySet(),
            dependsOnDescendants: Boolean = false,
            externalContextDependencies: Iterable<String> = emptySet(),
            data: ValidationIssueData = emptyMap(),
        ): LocatedValidationIssue =
            LocatedValidationIssue(
                AbsolutePath(path),
                code,
                severity,
                dependencies.map { Path(it) },
                dependsOnDescendants,
                externalContextDependencies,
                data,
            )

        /** Creates a [LocatedValidationIssue] given a [path] and an [issue]. */
        internal operator fun invoke(
            path: Path,
            validation: Validation<*>,
            issue: ValidationIssue,
        ): LocatedValidationIssue {
            val absolutePath = path.toAbsolutePath()
            val dependencies =
                resolveDependencies(absolutePath, validation.dependencies.values.map { it.path })
            return when (issue) {
                is ValidationError ->
                    LocatedValidationError(
                        absolutePath,
                        issue.code,
                        dependencies,
                        validation.dependsOnDescendants,
                        validation.externalContextDependencies,
                        issue.data,
                    )
                is ValidationWarning ->
                    LocatedValidationWarning(
                        absolutePath,
                        issue.code,
                        dependencies,
                        validation.dependsOnDescendants,
                        validation.externalContextDependencies,
                        issue.data,
                    )
                is ValidationFailure ->
                    LocatedValidationFailure(
                        absolutePath,
                        issue.code,
                        dependencies,
                        validation.dependsOnDescendants,
                        validation.externalContextDependencies,
                        issue.data,
                    )
            }
        }

        /**
         * Utility function that creates a set of absolute dependency paths given the issue path and
         * the collection of dependency paths.
         */
        internal fun resolveDependencies(
            path: Path,
            dependencies: Iterable<Path>,
        ): Set<AbsolutePath> =
            if (dependencies.none()) emptySet()
            else
                path.toAbsolutePath().let { absolutePath ->
                    dependencies.mapTo(mutableSetOf()) { absolutePath.resolve(it) }
                }
    }
}

/**
 * Validation error emitted by a [form validator][FormValidator] or [manager][FormManager]
 * containing location information.
 */
@JsName("LocatedValidationErrorKt")
@Serializable
@SerialName("error")
public class LocatedValidationError
@JvmOverloads
constructor(
    override val path: AbsolutePath,
    override val code: String,
    override val dependencies: Set<AbsolutePath> = emptySet(),
    override val dependsOnDescendants: Boolean = false,
    override val externalContextDependencies: Set<String> = emptySet(),
    override val data: ValidationIssueData = emptyMap(),
) : LocatedValidationIssue() {
    @JvmOverloads
    public constructor(
        path: Path,
        code: String,
        dependencies: Iterable<Path> = emptySet(),
        dependsOnDescendants: Boolean = false,
        externalContextDependencies: Iterable<String> = emptySet(),
        data: ValidationIssueData = emptyMap(),
    ) : this(
        AbsolutePath(path),
        code,
        resolveDependencies(path, dependencies),
        dependsOnDescendants,
        externalContextDependencies.toSet(),
        data,
    )

    @JvmOverloads
    public constructor(
        path: String,
        code: String,
        dependencies: Iterable<String> = emptySet(),
        dependsOnDescendants: Boolean = false,
        externalContextDependencies: Iterable<String> = emptySet(),
        data: ValidationIssueData = emptyMap(),
    ) : this(
        AbsolutePath(path),
        code,
        dependencies.map { Path(it) },
        dependsOnDescendants,
        externalContextDependencies,
        data,
    )

    override fun toString(): String = buildString {
        append("LocatedValidationError(path=$path, code=$code")
        if (dependencies.isNotEmpty()) {
            append(", dependencies=$dependencies")
        }
        if (dependsOnDescendants) {
            append(", dependsOnDescendants=true")
        }
        if (externalContextDependencies.isNotEmpty()) {
            append(", externalContextDependencies=$externalContextDependencies")
        }
        if (data.isNotEmpty()) {
            append(", data=$data")
        }
        append(")")
    }
}

/**
 * Validation warning emitted by a [form validator][FormValidator] or [manager][FormManager]
 * containing location information.
 */
@JsName("LocatedValidationWarningKt")
@Serializable
@SerialName("warning")
public class LocatedValidationWarning
@JvmOverloads
constructor(
    override val path: AbsolutePath,
    override val code: String,
    override val dependencies: Set<AbsolutePath> = emptySet(),
    override val dependsOnDescendants: Boolean = false,
    override val externalContextDependencies: Set<String> = emptySet(),
    override val data: ValidationIssueData = emptyMap(),
) : LocatedValidationIssue() {
    @JvmOverloads
    public constructor(
        path: Path,
        code: String,
        dependencies: Iterable<Path> = emptySet(),
        dependsOnDescendants: Boolean = false,
        externalContextDependencies: Iterable<String> = emptySet(),
        data: ValidationIssueData = emptyMap(),
    ) : this(
        AbsolutePath(path),
        code,
        resolveDependencies(path, dependencies),
        dependsOnDescendants,
        externalContextDependencies.toSet(),
        data,
    )

    @JvmOverloads
    public constructor(
        path: String,
        code: String,
        dependencies: Iterable<String> = emptySet(),
        dependsOnDescendants: Boolean = false,
        externalContextDependencies: Iterable<String> = emptySet(),
        data: ValidationIssueData = emptyMap(),
    ) : this(
        AbsolutePath(path),
        code,
        dependencies.map { Path(it) },
        dependsOnDescendants,
        externalContextDependencies,
        data,
    )

    override fun toString(): String = buildString {
        append("LocatedValidationWarning(path=$path, code=$code")
        if (dependencies.isNotEmpty()) {
            append(", dependencies=$dependencies")
        }
        if (dependsOnDescendants) {
            append(", dependsOnDescendants=true")
        }
        if (externalContextDependencies.isNotEmpty()) {
            append(", externalContextDependencies=$externalContextDependencies")
        }
        if (data.isNotEmpty()) {
            append(", data=$data")
        }
        append(")")
    }
}

/**
 * Validation error emitted by a [manager][FormManager] containing location information when a
 * validation fails to run (throws an exception while being executed). The code of this validation
 * error is [ValidationFailure.CODE].
 *
 * The validation error [data] contains entries with keys `"validation"`, `"exception"`, and
 * `"stackTrace"` containing information on the name of the validation that caused the exception as
 * well as information about the exception itself. These data properties are also accessible via
 * properties of the class with the same name.
 */
@JsName("LocatedValidationFailureKt")
@Serializable
@SerialName("failure")
public class LocatedValidationFailure
internal constructor(
    override val path: AbsolutePath,
    override val code: String,
    override val dependencies: Set<AbsolutePath> = emptySet(),
    override val dependsOnDescendants: Boolean = false,
    override val externalContextDependencies: Set<String> = emptySet(),
    override val data: ValidationIssueData = emptyMap(),
) : LocatedValidationIssue() {
    /** Name of the validation where the exception occurred. */
    public val validation: String
        get() = data["validation"]!!

    /** Exception that occurred (name and message). */
    public val exception: String?
        get() = data["exception"]

    /** Exception stack trace. */
    public val stackTrace: String
        get() = data["stackTrace"]!!

    override fun toString(): String = buildString {
        append("LocatedValidationFailure(path=$path, code=$code, validation=$validation")
        if (exception != null) {
            append(", exception=$exception")
        }
        if (dependencies.isNotEmpty()) {
            append(", dependencies=$dependencies")
        }
        if (dependsOnDescendants) {
            append(", dependsOnDescendants=true")
        }
        if (externalContextDependencies.isNotEmpty()) {
            append(", externalContextDependencies=$externalContextDependencies")
        }
        append(")")
    }
}

/** Returns whether a flow of located validation issues contains any issues at all. */
public suspend fun Flow<LocatedValidationIssue>.containsIssues(): Boolean = firstOrNull() != null

/** Returns whether a flow of located validation issues contains no issues at all. */
public suspend fun Flow<LocatedValidationIssue>.containsNoIssues(): Boolean = !containsIssues()

/** Returns whether a flow of located validation issues contains any errors. */
public suspend fun Flow<LocatedValidationIssue>.containsErrors(): Boolean =
    firstOrNull { issue -> issue.severity == ValidationIssueSeverity.Error } != null

/** Returns whether a flow of located validation issues contains no errors. */
public suspend fun Flow<LocatedValidationIssue>.containsNoErrors(): Boolean = !containsErrors()

/** Returns whether a flow of located validation issues contains any warnings. */
public suspend fun Flow<LocatedValidationIssue>.containsWarnings(): Boolean =
    firstOrNull { issue -> issue.severity == ValidationIssueSeverity.Warning } != null

/** Returns whether a flow of located validation issues contains no warnings. */
public suspend fun Flow<LocatedValidationIssue>.containsNoWarnings(): Boolean = !containsWarnings()

/** Returns whether a collection of located validation issues contains any issues at all. */
public fun Iterable<LocatedValidationIssue>.containsIssues(): Boolean = any()

/** Returns whether a collection of located validation issues contains no issues at all. */
public fun Iterable<LocatedValidationIssue>.containsNoIssues(): Boolean = !containsIssues()

/** Returns whether a collection of located validation issues contains any errors. */
public fun Iterable<LocatedValidationIssue>.containsErrors(): Boolean = any { issue ->
    issue.severity == ValidationIssueSeverity.Error
}

/** Returns whether a collection of located validation issues contains no errors. */
public fun Iterable<LocatedValidationIssue>.containsNoErrors(): Boolean = !containsErrors()

/** Returns whether a collection of located validation issues contains any warnings. */
public fun Iterable<LocatedValidationIssue>.containsWarnings(): Boolean = any { issue ->
    issue.severity == ValidationIssueSeverity.Warning
}

/** Returns whether a collection of located validation issues contains no warnings. */
public fun Iterable<LocatedValidationIssue>.containsNoWarnings(): Boolean = !containsWarnings()
