package tech.ostack.kform

/** Validation status of a value. */
public enum class ValidationStatus {
    /** Validation status representing that the value hasn't been validated. */
    Unvalidated,
    /** Validation status representing that the value is currently being validated. */
    Validating,
    /** Validation status representing that the value has been validated successfully. */
    Validated,
    /**
     * Validation status representing that the value has been validated, but an exception occurred
     * during the validation.
     */
    ValidatedExceptionally,
}

/**
 * Display status of a value.
 *
 * This status represents how a value should be displayed in terms of its validity, taking into
 * consideration whether it has been "touched". I.e. the value might be invalid but, because it
 * hasn't yet been touched, its display status would be [Valid].
 *
 * Furthermore, the display status of a value takes descendant values into consideration. This means
 * that a parent value containing a "touched" descendant with errors will have a display status of
 * [Error].
 */
public enum class DisplayStatus {
    /**
     * Display status representing that the value is either "unvalidated", "untouched", or that
     * itself and all of its descendants are valid.
     */
    Valid,
    /**
     * Display status representing that the value is "touched" and that either itself of one of its
     * descendants has an error.
     */
    Error,
    /**
     * Display status representing that the value is "touched", that either itself of one of its
     * descendants has a warning, and that neither itself nor any of its descendants has an error.
     */
    Warning,
}

/** Status of the automatic validations. */
public enum class AutoValidationStatus {
    /**
     * Automatic validations are inactive (validation mode is set to
     * [Manual][ValidationMode.Manual]).
     */
    Inactive,
    /**
     * Automatic validations are active but idle (validation mode is set to
     * [Auto][ValidationMode.Auto] but there is nothing to validate).
     */
    ActiveIdle,
    /**
     * Automatic validations are active and running (validation mode is set to
     * [Auto][ValidationMode.Auto] and values are currently being validated).
     */
    ActiveRunning,
}

/** Status of the computed values. */
public enum class ComputedValueStatus {
    Inactive,
    ActiveIdle,
    ActiveRunning,
}
