package tech.ostack.kform

import io.github.oshai.kotlinlogging.KotlinLogging
import kotlin.coroutines.CoroutineContext
import kotlin.js.JsName
import kotlin.jvm.JvmOverloads
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.toList
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import tech.ostack.kform.internal.*
import tech.ostack.kform.internal.actions.*

/** Function used to handle the flow of value information related to some path. */
public typealias ValueInfoHandler<TResult> = suspend (infoFlow: Flow<ValueInfo<*>>) -> TResult

/** Function used to handle the flow of information related to some path. */
public typealias InfoHandler<TResult> = suspend (infoFlow: Flow<Info<*>>) -> TResult

/** Function used to handle a gotten value. */
public typealias ValueHandler<T, TResult> = suspend (value: T) -> TResult

/** Function used to handle the flow of located validation issues. */
public typealias IssuesHandler<TResult> =
    suspend (issuesFlow: Flow<LocatedValidationIssue>) -> TResult

/** Function used to handle the gotten external context. */
public typealias ExternalContextHandler<T, TResult> = suspend (externalContext: T) -> TResult

/** Function used to subscribe to form manager events. */
public typealias EventHandler = suspend (event: FormManagerEvent<*>) -> Unit

/** Function to run during a form manager subscription. */
public typealias OnSubscription = suspend () -> Unit

/** Function used to unsubscribe from a form manager subscription. */
public typealias Unsubscribe = suspend () -> Unit

/** Form manager validation mode. */
public enum class ValidationMode {
    /** Automatic validations. A background process is launched that runs validations as needed. */
    Auto,
    /**
     * Manual validations. Validations only run when [FormManager.validate] (or
     * [FormManager.isValid]) is called.
     */
    Manual,
}

/**
 * Class responsible for managing the data and state of a form.
 *
 * The form manager stores, provides access, and allows manipulating the content of a form with a
 * provided [formSchema] in a concurrent and controlled manner. It further manages state associated
 * with the different fields of the form such as keeping track of validation issues, "dirty" and
 * "touched" states, and others.
 *
 * All form value/state accesses and manipulations should go through an instance of the form
 * manager, to make sure that changes are tracked and no "dangerous" concurrent operations are
 * performed. As such, storing non-copied form values outside the form manager is highly
 * discouraged. Certain methods like [valueInfo], [get], and [validate] take lambdas as arguments to
 * guarantee that data access only happens within a "controlled environment", i.e. during the
 * lifetime of the lambda.
 *
 * A `coroutineContext` may be provided to specify the context of coroutines launched by the form
 * manager.
 */
@JsName("FormManagerKt")
public class FormManager
@JvmOverloads
public constructor(
    internal val formSchema: Schema<*>,
    private val initialValue: Any?,
    externalContexts: ExternalContexts? = null,
    validationMode: ValidationMode = ValidationMode.Auto,
    coroutineContext: CoroutineContext = Dispatchers.Default,
    autoInit: Boolean = true,
) {
    @JvmOverloads
    public constructor(
        formSchema: Schema<*>,
        externalContexts: ExternalContexts? = null,
        validationMode: ValidationMode = ValidationMode.Auto,
        coroutineContext: CoroutineContext = Dispatchers.Default,
        autoInit: Boolean = true,
    ) : this(
        formSchema,
        INITIAL_VALUE,
        externalContexts,
        validationMode,
        coroutineContext,
        autoInit,
    )

    // Information on validation dependencies:
    internal val validationDependencies: ValidationDependencies =
        buildValidationDependencies(formSchema).also {
            logger.debug { "Validation dependencies: $it" }
        }
    internal val validationObservedDependencies: ObservedValidationDependencies =
        buildObservedValidationDependencies(formSchema).also {
            logger.debug { "Observed validation dependencies: $it" }
        }
    internal val externalContextValidationDependencies: ExternalContextValidationDependencies =
        buildExternalContextValidationDependencies(formSchema).also {
            logger.debug { "External context validation dependencies: $it" }
        }
    internal val externalIssuesDependencies: ExternalIssuesDependencies =
        ExternalIssuesDependencies()

    // Information on computed value dependencies:
    internal val computedValueDependencies: ComputedValueDependencies =
        buildComputedValueDependencies(formSchema).also {
            logger.debug { "Computed value dependencies: $it" }
        }
    internal val computedValueObservedDependencies: ObservedComputedValueDependencies =
        buildObservedComputedValueDependencies(formSchema).also {
            logger.debug { "Observed computed value dependencies: $it" }
        }
    internal val externalContextComputedValueDependencies:
        ExternalContextComputedValueDependencies =
        buildExternalContextComputedValueDependencies(formSchema).also {
            logger.debug { "External context computed value dependencies: $it" }
        }

    internal lateinit var supervisorJob: CompletableJob
    internal lateinit var scope: CoroutineScope

    private lateinit var actionManager: ActionManager
    internal lateinit var eventsBus: FormManagerEventsBus
    private val validationDaemon: ValidationDaemon = ValidationDaemon(this)

    // Global form manager state:
    internal var formValue: Any? = UNINITIALIZED
    internal var formState: State? = null
    internal var externalContexts: HashMap<String, Any?> = HashMap()

    // Initialisation state:
    private val initMutex = Mutex()
    private var initted = false // Whether [init] has run
    // Whether the form manager can be subscribed to
    private var subscribable: CompletableDeferred<Unit> = CompletableDeferred()
    // Whether the form manager is fully initialised (with completed initial action)
    private var initialized: CompletableDeferred<Unit> = CompletableDeferred()

    init {
        if (autoInit && initMutex.tryLock()) {
            syncInit(externalContexts, validationMode, coroutineContext)
            initMutex.unlock()
        }
    }

    private fun syncInit(
        externalContexts: ExternalContexts?,
        validationMode: ValidationMode,
        coroutineContext: CoroutineContext,
    ) {
        if (initted) return
        initted = true

        supervisorJob = SupervisorJob()
        scope = CoroutineScope(CoroutineName("Form manager") + supervisorJob + coroutineContext)
        actionManager = ActionManager(scope)
        eventsBus = FormManagerEventsBus()
        if (externalContexts != null) {
            this.externalContexts += externalContexts
        }
        subscribable.complete(Unit)

        // Initialise the form manager and, if required, the validation daemon; mark the form
        // manager as initialised once these initial actions succeed
        val completableInitialized = initialized
        scope
            .launch {
                scheduleActionAndAwait(SetAction(this@FormManager, null, null, null, initialValue))
                if (validationMode == ValidationMode.Auto) {
                    scheduleActionAndAwait(
                        StartValidationDaemonAction(this@FormManager, validationDaemon)
                    )
                }
                completableInitialized.complete(Unit)
                logger.debug { "Initialized form manager" }
            }
            .invokeOnCompletion { ex ->
                if (ex != null) {
                    completableInitialized.completeExceptionally(ex)
                    if (ex is CancellationException) {
                        logger.debug { "Cancelled initialization of form manager ($ex)" }
                    } else {
                        logger.error(ex) { "Failed to initialize form manager" }
                    }
                }
            }
    }

    /**
     * Initialises the form manager.
     *
     * This method is automatically called when constructing a new form manager instance when
     * `autoInit` is set to `true` (the default). Calling [init] on an already initted form manager
     * has no effect.
     */
    public suspend fun init(
        externalContexts: ExternalContexts? = null,
        validationMode: ValidationMode = ValidationMode.Auto,
        coroutineContext: CoroutineContext = Dispatchers.Default,
    ) {
        initMutex.withLock { syncInit(externalContexts, validationMode, coroutineContext) }
    }

    /**
     * Destroys this form manager instance by cancelling its coroutine scope.
     *
     * Destroying an already destroyed form manager has no effect.
     */
    public suspend fun destroy() {
        initMutex.withLock {
            if (!initted) return
            initted = false

            withContext(NonCancellable) {
                val cancellationException = CancellationException("Destroyed form manager.")
                subscribable.completeExceptionally(cancellationException)
                subscribable = CompletableDeferred()
                initialized.completeExceptionally(cancellationException)
                initialized = CompletableDeferred()
                scope.cancel(cancellationException)
                supervisorJob.join()
                validationDaemon.stop()
                formValue = UNINITIALIZED
                formState = null
                externalContexts.clear()
                logger.debug { "Destroyed form manager" }
            }
        }
    }

    /**
     * Awaits initialisation of the form manager: in practice, it waits for the initial `SetAction`
     * to be scheduled.
     */
    private suspend fun awaitInitialization() = initialized.await()

    /** Awaits for the form manager to become subscribable. */
    private suspend fun awaitSubscribable() = subscribable.await()

    /** Schedules an [action] to run and returns it without waiting for its resolution. */
    internal suspend fun <TResult, TAction : Action<TResult>> scheduleAction(
        action: TAction
    ): TAction = action.also { actionManager.scheduleAction(action) }

    /**
     * Schedules an [action] to run and awaits for its resolution, returning the result of the
     * action.
     */
    internal suspend fun <TResult, TAction : Action<TResult>> scheduleActionAndAwait(
        action: TAction
    ): TResult = scheduleAction(action).await()

    /** Status of the automatic validations. */
    public val autoValidationStatus: StateFlow<AutoValidationStatus>
        get() = validationDaemon.status

    /** Sets the validation mode. */
    public suspend fun setValidationMode(validationMode: ValidationMode) {
        awaitInitialization()
        scheduleActionAndAwait(
            if (validationMode == ValidationMode.Auto)
                StartValidationDaemonAction(this, validationDaemon)
            else StopValidationDaemonAction(this, validationDaemon)
        )
    }

    private fun schemaInfoImpl(path: AbsolutePath): Sequence<SchemaInfo<*>> =
        schemaInfoImpl(formSchema, path)

    // Throws when [path] doesn't match any schema path
    private fun validatePath(path: AbsolutePath) {
        if (!schemaInfoImpl(path).any()) {
            throw InvalidPathException(path, "No schema path matches this path.")
        }
    }

    /**
     * Returns whether there exists at least one schema at a path matching [path].
     *
     * Paths that match no schema paths are deemed invalid by the form manager and most methods
     * called with them will throw.
     */
    public fun isValidPath(path: Path): Boolean = schemaInfoImpl(path.toAbsolutePath()).any()

    /**
     * Returns whether there exists at least one schema at a path matching [path].
     *
     * Paths that match no schema paths are deemed invalid by the form manager and most methods
     * called with them will throw.
     */
    public fun isValidPath(path: String): Boolean = isValidPath(AbsolutePath(path))

    /**
     * Returns a sequence of information about the schemas at paths matching [path].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public fun schemaInfo(path: Path = AbsolutePath.MATCH_ALL): Sequence<SchemaInfo<*>> =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            schemaInfoImpl(absolutePath)
        }

    /**
     * Returns a sequence of information about the schemas at paths matching [path].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public fun schemaInfo(path: String): Sequence<SchemaInfo<*>> = schemaInfo(AbsolutePath(path))

    /**
     * Runs the [infoHandler] lambda with the value-information of values at paths matching [path].
     * Returns the result of [infoHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> valueInfo(
        path: Path = AbsolutePath.MATCH_ALL,
        @BuilderInference infoHandler: ValueInfoHandler<TResult>,
    ): TResult =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(ValueInfoAction(this, absolutePath, infoHandler))
        }

    /**
     * Runs the [infoHandler] lambda with the value-information of values at paths matching [path].
     * Returns the result of [infoHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> valueInfo(
        path: String,
        @BuilderInference infoHandler: ValueInfoHandler<TResult>,
    ): TResult = valueInfo(AbsolutePath(path), infoHandler)

    /**
     * Runs [infoHandler] with all information of values at paths matching [path]. Returns the
     * result of [infoHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> info(
        path: Path = AbsolutePath.MATCH_ALL,
        @BuilderInference infoHandler: InfoHandler<TResult>,
    ): TResult =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(InfoAction(this, absolutePath, infoHandler))
        }

    /**
     * Runs [infoHandler] with all information of values at paths matching [path]. Returns the
     * result of [infoHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> info(
        path: String,
        @BuilderInference infoHandler: InfoHandler<TResult>,
    ): TResult = info(AbsolutePath(path), infoHandler)

    /**
     * Returns the single schema matching [path].
     *
     * To get information about all schemas at paths matching a given path use [schemaInfo] instead.
     *
     * @throws InvalidPathException If [path] matches no schema paths or more than one schema path.
     */
    public fun schema(path: Path = AbsolutePath.ROOT): Schema<*> =
        path.toAbsolutePath().let { absolutePath ->
            try {
                schemaInfo(absolutePath).single().schema
            } catch (_: IllegalArgumentException) {
                throw InvalidPathException(absolutePath, "Path matches more than one schema path.")
            }
        }

    /**
     * Returns the single schema matching [path].
     *
     * To get information about all schemas at paths matching a given path use [schemaInfo] instead.
     *
     * @throws InvalidPathException If [path] matches no schema paths or more than one schema path.
     */
    public fun schema(path: String): Schema<*> = schema(AbsolutePath(path))

    /**
     * Returns whether there exists at least one value at a path matching [path].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun has(path: Path): Boolean =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(HasAction(this, absolutePath))
        }

    /**
     * Returns whether there exists at least one value at a path matching [path].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun has(path: String): Boolean = has(AbsolutePath(path))

    /**
     * Runs the provided [valueHandler] with the single value at [path]. Returns the result of
     * [valueHandler].
     *
     * Because this method is meant to return a single value, paths with wildcards are not accepted.
     * To get all values at paths matching a path containing wildcards, use [valueInfo] instead.
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] contains wildcards, matches no schemas, or matches
     *   more than one value.
     * @throws IllegalStateException If [path] matches more than one value.
     * @throws NoSuchElementException If no value matches [path].
     */
    public suspend fun <T, TResult> get(
        path: Path = AbsolutePath.ROOT,
        @BuilderInference valueHandler: ValueHandler<T, TResult>,
    ): TResult =
        path.toAbsolutePath().let { absolutePath ->
            if (absolutePath.hasAnyWildcard()) {
                throw InvalidPathException(absolutePath, "Path cannot contain wildcards.")
            }
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(GetAction(this, absolutePath, valueHandler))
        }

    /**
     * Runs the provided [valueHandler] with the single value at [path]. Returns the result of
     * [valueHandler].
     *
     * Because this method is meant to return a single value, paths with wildcards are not accepted.
     * To get all values at paths matching a path containing wildcards, use [valueInfo] instead.
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] contains wildcards, matches no schemas, or matches
     *   more than one value.
     * @throws IllegalStateException If [path] matches more than one value.
     * @throws NoSuchElementException If no value matches [path].
     */
    public suspend fun <T, TResult> get(
        path: String,
        @BuilderInference valueHandler: ValueHandler<T, TResult>,
    ): TResult = get(AbsolutePath(path), valueHandler)

    /**
     * Returns a clone (deep copy) of the single value at [path]. Equivalent to:
     * ```kotlin
     * get(path) { value -> schema(path).clone(value) }
     * ```
     *
     * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
     * @throws IllegalStateException If [path] matches more than one value.
     * @throws NoSuchElementException If no value matches [path].
     */
    public suspend fun <T> getClone(path: Path = AbsolutePath.ROOT): T =
        path.toAbsolutePath().let { absolutePath ->
            if (absolutePath.hasAnyWildcard()) {
                throw InvalidPathException(absolutePath, "Path cannot contain wildcards.")
            }
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(GetCloneAction(this, absolutePath))
        }

    /**
     * Returns a clone (deep copy) of the single value at [path]. Equivalent to:
     * ```kotlin
     * get(path) { value -> schema(path).clone(value) }
     * ```
     *
     * @throws InvalidPathException If [path] contains wildcards or matches no schemas.
     * @throws IllegalStateException If [path] matches more than one value.
     * @throws NoSuchElementException If no value matches [path].
     */
    public suspend fun <T> getClone(path: String): T = getClone(AbsolutePath(path))

    /**
     * Sets values at [path] with [toSet].
     *
     * If the path has a trailing non-recursive wildcard, then all existing children of its parent
     * value are set to [toSet]. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`; setting
     * the value `5` at `"/list/∗"` will cause `"/list"` to end up with `[5, 5, 5]`.
     *
     * Setting a value on a path with a trailing recursive wildcard is considered equivalent to
     * setting the value on said path without such wildcard. E.g. setting the value at `"/x/∗∗"` is
     * equivalent to setting the same value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun set(path: Path = AbsolutePath.ROOT, toSet: Any?): Unit =
        path.toAbsolutePath().let { absolutePath ->
            // Normalise the path by removing trailing recursive wildcards
            val normalizedPath =
                if (absolutePath.lastFragment is AbsolutePathFragment.RecursiveWildcard)
                    absolutePath.parent()
                else absolutePath

            validatePath(normalizedPath)
            awaitInitialization()

            // Schedule a set action per different targeted parent schema; knowing that an action is
            // being performed on a single parent schema allows for some optimisations (it allows us
            // to have the schema specify whether it supports concurrent operations; as a result,
            // for example, the class schema allows concurrent reads/writes to different members)
            val actions = mutableListOf<SetAction>()
            if (normalizedPath.isRoot) {
                actions += scheduleAction(SetAction(this, null, null, null, toSet))
            } else {
                val fragment = normalizedPath.lastFragment
                val parentPath = normalizedPath.parent()
                for ((parentSchema, _, parentQueriedPath) in schemaInfo(parentPath)) {
                    @Suppress("UNCHECKED_CAST") (parentSchema as ParentSchema<Any?>)
                    actions +=
                        scheduleAction(
                            SetAction(this, parentQueriedPath, parentSchema, fragment, toSet)
                        )
                }
            }

            for (action in actions) {
                try {
                    action.await()
                } catch (_: OverriddenActionException) {}
            }
        }

    /**
     * Sets values at [path] with [toSet].
     *
     * If the path has a trailing non-recursive wildcard, then all existing children of its parent
     * value are set to [toSet]. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`; setting
     * the value `5` at `"/list/∗"` will cause `"/list"` to end up with `[5, 5, 5]`.
     *
     * Setting a value on a path with a trailing recursive wildcard is considered equivalent to
     * setting the value on said path without such wildcard. E.g. setting the value at `"/x/∗∗"` is
     * equivalent to setting the same value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun set(path: String, toSet: Any?): Unit = set(AbsolutePath(path), toSet)

    /**
     * Resets the values at [path] to their initial value.
     *
     * If the path has a trailing non-recursive wildcard, then all existing children of its parent
     * value will have their value reset. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`
     * and that the schema of `"/list/∗"` has an initial value of `0`; resetting `"/list/∗"` will
     * thus cause `"/list"` to end up with `[0, 0, 0]`.
     *
     * Resetting the value on a path with a trailing recursive wildcard is considered equivalent to
     * resetting the value on said path without such wildcard. E.g. resetting the value at `"/x/∗∗"`
     * is equivalent to resetting the value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun reset(path: Path = AbsolutePath.ROOT): Unit = set(path, INITIAL_VALUE)

    /**
     * Resets the values at [path] to their initial value.
     *
     * If the path has a trailing non-recursive wildcard, then all existing children of its parent
     * value will have their value reset. E.g. assume that the list `[1, 2, 3]` exists at `"/list"`
     * and that the schema of `"/list/∗"` has an initial value of `0`; resetting `"/list/∗"` will
     * thus cause `"/list"` to end up with `[0, 0, 0]`.
     *
     * Resetting the value on a path with a trailing recursive wildcard is considered equivalent to
     * resetting the value on said path without such wildcard. E.g. resetting the value at `"/x/∗∗"`
     * is equivalent to resetting the value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun reset(path: String): Unit = reset(AbsolutePath(path))

    /**
     * Removes the values matching [path] from their parent collection(s).
     *
     * It is possible to clear a collection by providing a path with a trailing wildcard.
     *
     * Removing the value on a path with a trailing recursive wildcard is considered equivalent to
     * removing the value on said path without such wildcard. E.g. removing the value at `"/x/∗∗"`
     * is equivalent to removing the value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths, when attempting to remove the
     *   root value, or when a parent of [path] is not a collection.
     */
    public suspend fun remove(path: Path): Unit =
        path.toAbsolutePath().let { absolutePath ->
            // Normalise the path by removing trailing recursive wildcards
            val normalizedPath =
                if (absolutePath.lastFragment is AbsolutePathFragment.RecursiveWildcard)
                    absolutePath.parent()
                else absolutePath

            if (normalizedPath.isRoot) {
                throw InvalidPathException(normalizedPath, "Cannot remove root value.")
            }
            validatePath(normalizedPath)
            val parentPath = normalizedPath.parent()
            val parentSchemasInfo = schemaInfo(parentPath)
            for ((parentSchema, _, parentQueriedPath) in parentSchemasInfo) {
                if (parentSchema !is CollectionSchema<*, *>) {
                    throw InvalidPathException(
                        normalizedPath,
                        "Schema at '$parentQueriedPath' is not a collection schema.",
                    )
                }
            }
            awaitInitialization()

            // Issue a remove action per different targeted parent schema (same reason as written
            // within [set]).
            val actions = mutableListOf<RemoveAction>()
            val fragment = normalizedPath.lastFragment!!
            for ((parentSchema, _, parentQueriedPath) in parentSchemasInfo) {
                @Suppress("UNCHECKED_CAST") (parentSchema as CollectionSchema<Any?, Any?>)
                actions +=
                    scheduleAction(RemoveAction(this, parentQueriedPath, parentSchema, fragment))
            }

            for (action in actions) {
                try {
                    action.await()
                } catch (_: OverriddenActionException) {}
            }
        }

    /**
     * Removes the values matching [path] from their parent collection(s).
     *
     * It is possible to clear a collection by providing a path with a trailing wildcard.
     *
     * Removing the value on a path with a trailing recursive wildcard is considered equivalent to
     * removing the value on said path without such wildcard. E.g. removing the value at `"/x/∗∗"`
     * is equivalent to removing the value at `"/x"`.
     *
     * @throws InvalidPathException If [path] matches no schema paths, when attempting to remove the
     *   root value, or when a parent of [path] is not a collection.
     */
    public suspend fun remove(path: String): Unit = remove(AbsolutePath(path))

    /**
     * Runs [externalContextHandler] with the external context named [externalContextName] currently
     * available to validations.
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     */
    public suspend fun <T, TResult> getExternalContext(
        externalContextName: String,
        @BuilderInference externalContextHandler: ExternalContextHandler<T, TResult>,
    ): TResult =
        scheduleActionAndAwait(
            GetExternalContextAction(this, externalContextName, externalContextHandler)
        )

    /**
     * Sets an [external context][externalContext] with name [externalContextName] to be available
     * to validations and returns the previous external context associated with the same name if one
     * existed.
     */
    public suspend fun <T> setExternalContext(externalContextName: String, externalContext: T): T? =
        scheduleActionAndAwait(SetExternalContextAction(this, externalContextName, externalContext))

    /**
     * Removes the external context with name [externalContextName] available to validations and
     * returns it if it existed.
     */
    public suspend fun <T> removeExternalContext(externalContextName: String): T? =
        scheduleActionAndAwait(RemoveExternalContextAction(this, externalContextName))

    /**
     * Validates all values at paths matching [path] by running a function [issuesHandler] with the
     * flow of all found [validation issues][LocatedValidationIssue]. Returns the result of
     * [issuesHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> validate(
        path: Path = AbsolutePath.MATCH_ALL,
        @BuilderInference issuesHandler: IssuesHandler<TResult>,
    ): TResult =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(ValidateAction(this, absolutePath, issuesHandler))
        }

    /**
     * Validates all values at paths matching [path] by running a function [issuesHandler] with the
     * flow of all found [validation issues][LocatedValidationIssue]. Returns the result of
     * [issuesHandler].
     *
     * This method receives a lambda to ensure that no conflicting concurrent operations occur
     * during the lifetime of said lambda.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun <TResult> validate(
        path: String,
        @BuilderInference issuesHandler: IssuesHandler<TResult>,
    ): TResult = validate(AbsolutePath(path), issuesHandler)

    /**
     * Validates all values at paths matching [path] and returns a list of all found
     * [validation issues][LocatedValidationIssue].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun validate(path: Path = AbsolutePath.MATCH_ALL): List<LocatedValidationIssue> =
        validate(path) { issues: Flow<LocatedValidationIssue> -> issues.toList() }

    /**
     * Validates all values at paths matching [path] and returns a list of all found
     * [validation issues][LocatedValidationIssue].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun validate(path: String): List<LocatedValidationIssue> =
        validate(AbsolutePath(path))

    /**
     * Returns whether the values at paths matching [path] are valid according to their schemas.
     *
     * Values are said to be valid if they contain no validation errors.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isValid(path: Path = AbsolutePath.MATCH_ALL): Boolean =
        validate(path) { issues -> issues.containsNoErrors() }

    /**
     * Returns whether the values at paths matching [path] are valid according to their schemas.
     *
     * Values are said to be valid if they contain no validation errors.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isValid(path: String): Boolean = isValid(AbsolutePath(path))

    /**
     * Adds external issues to the form manager. Once added, each issue can be removed either
     * manually via [removeExternalIssues] or automatically when the value referenced by the issue
     * or one of the values referenced by the issue's dependencies change.
     *
     * @throws InvalidPathException If any of the issue's paths match no schema paths.
     */
    public suspend fun addExternalIssues(issues: Iterable<LocatedValidationIssue>) {
        for (issue in issues) {
            validatePath(issue.path)
            for (dependency in issue.dependencies) {
                validatePath(dependency)
            }
        }
        awaitInitialization()

        scheduleActionAndAwait(AddExternalIssuesAction(this, issues))
    }

    /**
     * Removes all external issues currently added to the form manager with paths matching [path].
     * If a [code] is provided, only the issues with the provided code are removed.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun removeExternalIssues(
        path: Path = AbsolutePath.MATCH_ALL,
        code: String? = null,
    ): Unit =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(RemoveExternalIssuesAction(this, absolutePath, code))
        }

    /**
     * Removes all external issues currently added to the form manager with paths matching [path].
     * If a [code] is provided, only the issues with the provided code are removed.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun removeExternalIssues(path: String, code: String? = null): Unit =
        removeExternalIssues(AbsolutePath(path), code)

    /**
     * Returns whether at least one value at a path matching [path] is dirty.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isDirty(path: Path = AbsolutePath.ROOT): Boolean =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(IsDirtyAction(this, absolutePath))
        }

    /**
     * Returns whether at least one value at a path matching [path] is dirty.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isDirty(path: String): Boolean = isDirty(AbsolutePath(path))

    /**
     * Returns whether all values at a path matching [path] are pristine (i.e. are not dirty).
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isPristine(path: Path = AbsolutePath.ROOT): Boolean = !isDirty(path)

    /**
     * Returns whether all values at a path matching [path] are pristine (i.e. are not dirty).
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isPristine(path: String): Boolean = isPristine(AbsolutePath(path))

    /**
     * Sets all values at a path matching [path] as dirty, as well as their parents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setDirty(path: Path = AbsolutePath.MATCH_ALL): Unit =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(SetDirtyAction(this, absolutePath))
        }

    /**
     * Sets all values at a path matching [path] as dirty, as well as their parents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setDirty(path: String): Unit = setDirty(AbsolutePath(path))

    /**
     * Sets all values at paths matching [path] as pristine (i.e. as not dirty), as well as their
     * descendents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setPristine(path: Path = AbsolutePath.ROOT): Unit =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(SetPristineAction(this, absolutePath))
        }

    /**
     * Sets all values at paths matching [path] as pristine (i.e. as not dirty), as well as their
     * descendents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setPristine(path: String): Unit = setPristine(AbsolutePath(path))

    /**
     * Returns whether at least one value at a path matching [path] has been touched.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isTouched(path: Path = AbsolutePath.ROOT): Boolean =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(IsTouchedAction(this, absolutePath))
        }

    /**
     * Returns whether at least one value at a path matching [path] has been touched.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isTouched(path: String): Boolean = isTouched(AbsolutePath(path))

    /**
     * Returns whether all values at paths matching [path] are untouched (i.e. are not touched).
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isUntouched(path: Path = AbsolutePath.ROOT): Boolean = !isTouched(path)

    /**
     * Returns whether all values at paths matching [path] are untouched (i.e. are not touched).
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun isUntouched(path: String): Boolean = isUntouched(AbsolutePath(path))

    /**
     * Sets all values at paths matching [path] as touched, as well as their parents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setTouched(path: Path = AbsolutePath.MATCH_ALL): Unit =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(SetTouchedAction(this, absolutePath))
        }

    /**
     * Sets all values at paths matching [path] as touched, as well as their parents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setTouched(path: String): Unit = setTouched(AbsolutePath(path))

    /**
     * Sets all values at paths matching [path] as untouched (i.e. as not touched), as well as their
     * descendents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setUntouched(path: Path = AbsolutePath.ROOT): Unit =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitInitialization()

            scheduleActionAndAwait(SetUntouchedAction(this, absolutePath))
        }

    /**
     * Sets all values at paths matching [path] as untouched (i.e. as not touched), as well as their
     * descendents.
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun setUntouched(path: String): Unit = setUntouched(AbsolutePath(path))

    /**
     * Subscribes to all events with paths matching [path] by running [eventHandler] for each event.
     * Returns a function that should be called to unsubscribe from the subscription.
     *
     * An [onSubscription] function may be provided, which is guaranteed to run after the
     * subscription has completed but before any events are emitted to the [eventHandler].
     *
     * All subscriptions are automatically cancelled when the form manager is [destroyed][destroy].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun subscribe(
        path: Path = AbsolutePath.MATCH_ALL,
        onSubscription: OnSubscription? = null,
        eventHandler: EventHandler,
    ): Unsubscribe =
        path.toAbsolutePath().let { absolutePath ->
            validatePath(absolutePath)
            awaitSubscribable()

            eventsBus.subscribe(absolutePath, onSubscription, eventHandler)
        }

    /**
     * Subscribes to all events with paths matching [path] by running [eventHandler] for each event.
     * Returns a function that should be called to unsubscribe from the subscription.
     *
     * An [onSubscription] function may be provided, which is guaranteed to run after the
     * subscription has completed but before any events are emitted to the [eventHandler].
     *
     * All subscriptions are automatically cancelled when the form manager is [destroyed][destroy].
     *
     * @throws InvalidPathException If [path] matches no schema paths.
     */
    public suspend fun subscribe(
        path: String,
        onSubscription: OnSubscription? = null,
        eventHandler: EventHandler,
    ): Unsubscribe = subscribe(AbsolutePath(path), onSubscription, eventHandler)

    public companion object {
        /** Logger used by the form manager (and internal classes used by it). */
        internal val logger = KotlinLogging.logger {}
    }
}
