package tech.ostack.kform

/**
 * Event emitted by the form manager. Events can be of one of two "main" types: [ValueEvent]
 * (signaling a change in a value of the form) or [StateEvent] (signaling a change in the form's
 * state).
 */
public sealed interface FormManagerEvent<T> {
    /** Path associated with the event. */
    public val path: AbsolutePath
    /** Schema of the value associated with the event. */
    public val schema: Schema<T>
}

/** Event emitted by the form manager signaling a change in a value of type [T] of the form. */
public sealed class ValueEvent<T>(override val path: AbsolutePath, override val schema: Schema<T>) :
    FormManagerEvent<T> {
    /** Value before the event occurred. */
    public abstract val oldValue: T?

    /** Value after the event occurred, as found in the [form manager][FormManager]. */
    public abstract val value: T?

    /**
     * Event that indicates the initialisation of a value of type [T].
     *
     * @property value Initialised value.
     */
    public data class Init<T>(
        override val value: T,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : ValueEvent<T>(path, schema) {
        override val oldValue: Nothing? = null

        override fun toString(): String = "Init(path=$path, value=$value)"
    }

    /**
     * Event that indicates that a certain value of type [T] has changed.
     *
     * @property value Value after the change, as found in the [form manager][FormManager].
     * @property oldValue Old [form manager][FormManager] value.
     */
    public data class Change<T>(
        override val oldValue: T,
        override val value: T,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : ValueEvent<T>(path, schema) {
        override fun toString(): String = "Change(path=$path, oldValue=$oldValue, value=$value)"
    }

    /**
     * Event that indicates that a value of type [T] has been destroyed.
     *
     * @property value Value destroyed from the [form manager][FormManager].
     */
    public data class Destroy<T>(
        override val oldValue: T,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : ValueEvent<T>(path, schema) {
        override val value: Nothing? = null

        override fun toString(): String = "Destroy(path=$path, oldValue=$oldValue)"
    }

    /**
     * Event that indicates the addition of elements of type [TChildren] to a collection of type
     * [T].
     *
     * @property value Collection, as found in the [form manager][FormManager], after the addition
     *   of the element.
     * @property addedValue Added value.
     * @property id Identifier of added value.
     */
    public data class Add<T, TChildren>(
        override val value: T,
        val addedValue: TChildren,
        val id: AbsolutePathFragment.Id,
        override val path: AbsolutePath,
        override val schema: CollectionSchema<T, TChildren>,
    ) : ValueEvent<T>(path, schema) {
        override val oldValue: Nothing? = null

        override fun toString(): String = "Add(path=$path, addedValue=$addedValue, id=$id)"
    }

    /**
     * Event that indicates the removal of elements of type [TChildren] from a collection of type
     * [T].
     *
     * @property value Collection, as found in the [form manager][FormManager], after the removal of
     *   the element.
     * @property removedValue Removed value.
     * @property id Identifier of removed value.
     */
    public data class Remove<T, TChildren>(
        override val value: T,
        val removedValue: TChildren,
        val id: AbsolutePathFragment.Id,
        override val path: AbsolutePath,
        override val schema: CollectionSchema<T, TChildren>,
    ) : ValueEvent<T>(path, schema) {
        override val oldValue: Nothing? = null

        override fun toString(): String = "Remove(path=$path, removedValue=$removedValue, id=$id)"
    }
}

/** Event emitted by the form manager signaling a change in the form's state. */
public sealed class StateEvent<T>(override val path: AbsolutePath, override val schema: Schema<T>) :
    FormManagerEvent<T> {
    /**
     * Event emitted when the validation status of a value has changed.
     *
     * @property status New validation status of the value.
     * @property issues Issues of the value.
     */
    public data class ValidationChange<T>(
        val status: ValidationStatus,
        val issues: List<ValidationIssue>,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : StateEvent<T>(path, schema) {
        override fun toString(): String =
            "ValidationChange(path=$path, status=$status, issues=$issues)"
    }

    /**
     * Event emitted when the display status of a value has changed.
     *
     * @property status New display status.
     */
    public data class DisplayChange<T>(
        val status: DisplayStatus,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : StateEvent<T>(path, schema) {
        override fun toString(): String = "DisplayChange(path=$path, status=$status)"
    }

    /**
     * Event emitted when the dirty status of a value has changed.
     *
     * @property status New dirty status (`true` if dirty, `false` if not).
     */
    public data class DirtyChange<T>(
        val status: Boolean,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : StateEvent<T>(path, schema) {
        override fun toString(): String = "DirtyChange(path=$path, status=$status)"
    }

    /**
     * Event emitted when the touched status of a value has changed.
     *
     * @property status New touched status (`true` if touched, `false` if not).
     */
    public data class TouchedChange<T>(
        val status: Boolean,
        override val path: AbsolutePath,
        override val schema: Schema<T>,
    ) : StateEvent<T>(path, schema) {
        override fun toString(): String = "TouchedChange(path=$path, status=$status)"
    }
}
