package tech.ostack.kform

import kotlin.reflect.KType

/** Value information of a validation's dependencies. */
public typealias DependenciesValueInfo = Map<String, ValueInfo<*>?>

/**
 * Context provided to a [form manager][FormManager] computation when running it, contains the
 * values of the computation's dependencies at the time the computation was executed.
 */
public abstract class ComputationContext(
    private val schema: Schema<*>,
    /** Path at which the computation is occurring. */
    public val path: AbsolutePath,
    /** Path of the schema at which the computation is occurring. */
    public val schemaPath: AbsolutePath,
    /**
     * Value information for each dependency of the computation. Mapping of keys as declared in the
     * [computation dependencies][Computation.dependencies] to their respective value information.
     */
    public val dependenciesInfo: DependenciesValueInfo,
    public val externalContexts: ExternalContexts,
) {
    // internal var invalidateAfter: Duration? = null

    /** Schema at which the computation is occurring. */
    @Suppress("UNCHECKED_CAST") public fun <T> schema(): Schema<T> = schema as Schema<T>

    /**
     * Obtains the value information of the dependency with key [dependencyKey] or `null` if the
     * dependency could not be found in the form.
     *
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    @Suppress("UNCHECKED_CAST")
    public fun <T> dependencyInfoOrNull(dependencyKey: String): ValueInfo<T>? {
        require(dependencyKey in dependenciesInfo) {
            "No dependency found with key '$dependencyKey'."
        }
        return dependenciesInfo[dependencyKey] as ValueInfo<T>?
    }

    /**
     * Obtains the value information of the dependency with key [dependencyKey].
     *
     * @throws DependencyNotFoundException If the dependency could not be found in the form.
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun <T> dependencyInfo(dependencyKey: String): ValueInfo<T> =
        dependencyInfoOrNull(dependencyKey) ?: throw DependencyNotFoundException(dependencyKey)

    /**
     * Returns the path of the dependency with key [dependencyKey] or `null` if the dependency could
     * not be found in the form.
     *
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun dependencyPathOrNull(dependencyKey: String): AbsolutePath? =
        dependencyInfoOrNull<Any?>(dependencyKey)?.path

    /**
     * Returns the path of the dependency with key [dependencyKey].
     *
     * @throws DependencyNotFoundException If the dependency could not be found in the form.
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun dependencyPath(dependencyKey: String): AbsolutePath =
        dependencyInfo<Any?>(dependencyKey).path

    /**
     * Returns the schema of the dependency with key [dependencyKey] or `null` if the dependency
     * could not be found in the form.
     *
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun <T> dependencySchemaOrNull(dependencyKey: String): Schema<T>? =
        dependencyInfoOrNull<T>(dependencyKey)?.schema

    /**
     * Returns the schema of the dependency with key [dependencyKey].
     *
     * @throws DependencyNotFoundException If the dependency could not be found in the form.
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun <T> dependencySchema(dependencyKey: String): Schema<T> =
        dependencyInfo<T>(dependencyKey).schema

    /**
     * Returns the value of the dependency with key [dependencyKey] or `null` if the dependency
     * could not be found in the form.
     *
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun <T> dependencyOrNull(dependencyKey: String): T? =
        dependencyInfoOrNull<T>(dependencyKey)?.value

    /**
     * Returns the value of the dependency with key [dependencyKey].
     *
     * @throws DependencyNotFoundException If the dependency could not be found in the form.
     * @throws IllegalArgumentException If no dependency with key [dependencyKey] was defined in the
     *   computation.
     */
    public fun <T> dependency(dependencyKey: String): T = dependencyInfo<T>(dependencyKey).value

    /**
     * Returns the external context with name [externalContextName] available to the computation or
     * `null` if the external context could not be found.
     *
     * @throws IllegalArgumentException If no external context with key [externalContextName] was
     *   defined in the computation.
     */
    @Suppress("UNCHECKED_CAST")
    public fun <T> externalContextOrNull(externalContextName: String): T? {
        require(externalContextName in externalContexts) {
            "No external context found with name '$externalContextName'."
        }
        return externalContexts[externalContextName] as T?
    }

    /**
     * Returns the external context with name [externalContextName] available to the computation.
     *
     * @throws ExternalContextNotFoundException If the external context could not be found during
     *   the computation.
     * @throws IllegalArgumentException If no external context with key [externalContextName] was
     *   defined in the computation.
     */
    public fun <T> externalContext(externalContextName: String): T =
        externalContextOrNull(externalContextName)
            ?: throw ExternalContextNotFoundException(externalContextName)

    // TODO: Implement something like this
    // private fun invalidateAfter(duration: Duration) {
    //     require(duration >= Duration.ZERO) { "Duration must not be negative." }
    //     invalidateAfter = duration
    // }
}

/** Information about a validation dependency, includes its [path] and [type]. */
public data class DependencyInfo(val path: Path, val type: KType)

/**
 * [Form manager][FormManager] computation.
 *
 * Computations depend on [FormManager] values. These dependencies should be defined in the
 * [dependencies] property.
 */
public interface Computation {
    /**
     * Dependencies of the computation. Mapping of keys to the paths this computation depends on.
     * Keys can be used within a [ComputationContext] to access the value of the dependencies.
     *
     * Computation dependency paths may contain a single recursive wildcard at the end, indicating
     * that the computation should be reevaluated by the [form manager][FormManager] whenever a
     * child value of the dependency changes. Otherwise, a dependency must contain no wildcards.
     */
    public val dependencies: Map<String, DependencyInfo>

    /**
     * Set of external context dependencies of the computation.
     *
     * Each entry of this set represents the name of the external context being depended upon.
     */
    public val externalContextDependencies: Set<String>
}

/**
 * Function used to update the state of a stateful computation.
 *
 * This function is used to update the computation's state given the previous `state` and the
 * `event` whose path matches the path being observed.
 */
public typealias UpdateStateFn<T, TState> = suspend (state: TState, event: ValueEvent<T>) -> TState

/**
 * Observer used within a stateful computation.
 *
 * Defines a path to observe and a function to update the state of the computation whenever an event
 * with a path matching the one being observed occurs.
 */
public open class Observer<T, TState>(
    /** Path to observe. */
    public val toObserve: Path,

    /**
     * Updates the computation's state given the previous state and the event whose path matches the
     * path being observed.
     *
     * This function must return the new computation state which, unless it hasn't changed, should
     * be different ([equals]-wise) from the previous computation state (as such, when using an
     * object as state, a new instance should be returned when the state is updated). If the new
     * state differs from the old one (using [equals]), a re-computation will be scheduled.
     */
    public val updateState: UpdateStateFn<T, TState>,
)

/**
 * Computation of values where the [form manager][FormManager] maintains a state of type [TState]
 * for each managed computation.
 *
 * Stateful computations are useful when they require expensive operations over data and if it is
 * possible to save the result of such expensive operation and tweak it as data changes instead of
 * running the expensive operation all over again.
 */
public interface StatefulComputation<TContext : ComputationContext, TState> : Computation {
    /**
     * List of observers.
     *
     * Each observer contains a path to be observed and a function to update the state of the
     * computation, which will be called whenever an event with a path matching the one being
     * observed occurs.
     *
     * If an event path matches multiple observers, only the **first** observer will be called.
     */
    public val observers: List<Observer<Any?, TState>>

    /**
     * Initialises and returns the computation's state, within a [TContext] containing the values of
     * all declared dependencies.
     *
     * This method is called whenever a new value that needs to be computed is initialised.
     */
    public suspend fun TContext.initState(): TState

    /**
     * Destroys the computation's [state].
     *
     * Called by the [form manager][FormManager] whenever a state is no longer needed (i.e. when a
     * newer state has been produced or when the computation has been destroyed).
     */
    public suspend fun destroyState(state: TState)
}
