import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.isActive
import tech.ostack.kform.*
import tech.ostack.kform.schemas.ClassState
import tech.ostack.kform.schemas.util.commonRestrictions

internal class ObjectSchemaJs(
    val fieldsSchemas: Map<String, Schema<Any?>>,
    override val validations: List<Validation<Any>> = emptyList(),
    initialValue: Any? = null,
) : ParentSchema<Any> {
    override val typeInfo: TypeInfo =
        TypeInfo("Object", restrictions = commonRestrictions(validations))

    override val supportsConcurrentSets: Boolean = true

    override val initialValue: Any =
        initialValue
            ?: run {
                val args = LinkedHashMap<String, Any?>(fieldsSchemas.size)
                for ((field, schema) in fieldsSchemas) {
                    args[field] = schema.initialValue
                }
                newObject(args)
            }

    override suspend fun clone(value: Any): Any {
        val fields = LinkedHashMap<String, Any?>(fieldsSchemas.size)
        for ((field, schema) in fieldsSchemas) {
            fields[field] = schema.clone(value.asDynamic()[field])
        }
        return newObject(fields)
    }

    override fun isValidChildSchemaFragment(fragment: AbsolutePathFragment): Boolean =
        fragment is AbsolutePathFragment.Id && fieldsSchemas.containsKey(fragment.id)

    override fun childrenSchemas(
        path: AbsolutePath,
        queriedPath: AbsolutePath,
        fragment: AbsolutePathFragment,
    ): Sequence<SchemaInfo<*>> = sequence {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((field, schema) in fieldsSchemas) {
                yield(
                    SchemaInfo(
                        schema,
                        path.append(AbsolutePathFragment.Id(field)),
                        queriedPath.append(AbsolutePathFragment.Id(field)),
                    )
                )
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val fieldSchema = fieldsSchemas[fragment.id] ?: error("Invalid fragment '$fragment'.")
            yield(SchemaInfo(fieldSchema, path.append(fragment), queriedPath.append(fragment)))
        }
    }

    override suspend fun isValidChildFragment(value: Any, fragment: AbsolutePathFragment): Boolean =
        isValidChildSchemaFragment(fragment)

    override fun children(
        path: AbsolutePath,
        schemaPath: AbsolutePath,
        value: Any,
        fragment: AbsolutePathFragment,
    ): Flow<ValueInfo<Any?>> = flow {
        if (fragment is AbsolutePathFragment.Wildcard) {
            for ((field, schema) in fieldsSchemas) {
                val fieldId = AbsolutePathFragment.Id(field)
                emit(
                    ValueInfo(
                        value.asDynamic()[field],
                        schema,
                        path.append(fieldId),
                        schemaPath.append(fieldId),
                    )
                )
            }
        } else {
            fragment as AbsolutePathFragment.Id
            val field = fragment.id
            val fieldSchema = fieldsSchemas[field] ?: error("Invalid fragment '$fragment'.")
            emit(
                ValueInfo(
                    value.asDynamic()[field],
                    fieldSchema,
                    path.append(fragment),
                    schemaPath.append(fragment),
                )
            )
        }
    }

    override suspend fun init(
        path: AbsolutePath,
        fromValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: Any) -> Unit,
    ) {
        fromValue ?: error("Cannot initialise value from '$fromValue'.")
        val fields = LinkedHashMap<String, Any?>(fieldsSchemas.size)
        for ((field, schema) in fieldsSchemas) {
            schema.init(
                path.append(AbsolutePathFragment.Id(field)),
                fromValue.asDynamic()[field],
                eventsBus,
            ) {
                fields[field] = it
            }
        }
        val newValue = newObject(fields)
        setValue(newValue)
        eventsBus.emit(ValueEvent.Init(newValue, path, this))
    }

    override suspend fun change(
        path: AbsolutePath,
        value: Any,
        intoValue: Any?,
        eventsBus: SchemaEventsBus,
        setValue: suspend (value: Any) -> Unit,
    ) {
        intoValue ?: error("Cannot initialise value from '$intoValue'.")
        for ((field, schema) in fieldsSchemas) {
            if (!currentCoroutineContext().isActive) break // Don't do more work than necessary
            schema.change(
                path.append(AbsolutePathFragment.Id(field)),
                value.asDynamic()[field],
                intoValue.asDynamic()[field],
                eventsBus,
            ) {
                value.asDynamic()[field] = it
            }
        }
        setValue(value)
    }

    override suspend fun destroy(
        path: AbsolutePath,
        value: Any,
        eventsBus: SchemaEventsBus,
        removeValue: suspend (value: Any) -> Unit,
    ) {
        removeValue(value)
        eventsBus.emit(ValueEvent.Destroy(value, path, this))
        for ((field, schema) in fieldsSchemas) {
            schema.destroy(
                path.append(AbsolutePathFragment.Id(field)),
                value.asDynamic()[field],
                eventsBus,
            ) {}
        }
    }

    override suspend fun isValidSetFragment(value: Any, fragment: AbsolutePathFragment): Boolean =
        isValidChildFragment(value, fragment)

    override suspend fun set(
        path: AbsolutePath,
        value: Any,
        fragment: AbsolutePathFragment,
        childValue: Any?,
        eventsBus: SchemaEventsBus,
    ) {
        fragment as AbsolutePathFragment.Id
        val field = fragment.id
        val fieldSchema = fieldsSchemas[field] ?: error("Invalid fragment '$fragment'.")
        fieldSchema.change(path.append(fragment), value.asDynamic()[field], childValue, eventsBus) {
            value.asDynamic()[field] = it
        }
    }

    override fun childrenStatesContainer(): ParentState = ClassState(fieldsSchemas)

    /**
     * Returns a new object represented by this schema given a map of [fields], mapping the name of
     * the field to its value.
     */
    private fun newObject(fields: Map<String, Any?>): Any {
        val obj = emptyJsObject<Any?>()
        for ((field, value) in fields) {
            obj.asDynamic()[field] = value
        }
        return obj
    }
}

/** Schema representing a JavaScript object. */
@JsName("objectSchema")
@JsExport
public fun objectSchemaJs(
    optionsOrFieldsSchemas: RecordTs<String, Any>,
    fieldsSchemas: RecordTs<String, SchemaJs<Any?>>? = null,
): SchemaJs<Any> =
    @Suppress("UNCHECKED_CAST", "UNCHECKED_CAST_TO_EXTERNAL_INTERFACE")
    when (fieldsSchemas) {
        null ->
            ObjectSchemaJs(
                jsObjectToMap(optionsOrFieldsSchemas as RecordTs<String, SchemaJs<Any?>>) {
                    it.schemaKt
                }!!
            )
        else -> {
            val options = optionsOrFieldsSchemas as SchemaOptionsJs<Any>
            ObjectSchemaJs(
                jsObjectToMap(fieldsSchemas) { it.schemaKt }!!,
                options.validations?.map { it.validationKt } ?: emptyList(),
                options.initialValue,
            )
        }
    }.cachedToJs()
