import tech.ostack.kform.*

/**
 * [Validation issue severity][ValidationIssueSeverity] representation for use from JavaScript
 * (`"error" | "warning"`).
 */
public typealias ValidationIssueSeverityJs = String

internal fun ValidationIssueSeverityJs.toValidationIssueSeverityKt(): ValidationIssueSeverity =
    when (this) {
        "error" -> ValidationIssueSeverity.Error
        "warning" -> ValidationIssueSeverity.Warning
        else -> error("Invalid validation issue severity: '$this'.")
    }

internal fun ValidationIssueSeverity.toJs(): ValidationIssueSeverityJs =
    toString().replaceFirstChar { it.lowercase() }

/** [Validation issue][ValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationIssue")
public sealed class ValidationIssueJs {
    internal abstract val issueKt: ValidationIssue

    public val code: String
        get() = issueKt.code

    public val data: RecordTs<String, String?>
        get() = issueKt.data.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is ValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: ValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

/** [Validation error][ValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationError")
public class ValidationErrorJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationError =
        when (issueKtOrCode) {
            is ValidationError -> issueKtOrCode
            else -> ValidationError(issueKtOrCode.toString(), jsObjectToMap(data) ?: emptyMap())
        }
}

/** [Validation warning][ValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationWarning")
public class ValidationWarningJs(issueKtOrCode: Any, data: RecordTs<String, String?>? = null) :
    ValidationIssueJs() {
    override val issueKt: ValidationWarning =
        when (issueKtOrCode) {
            is ValidationWarning -> issueKtOrCode
            else -> ValidationWarning(issueKtOrCode.toString(), jsObjectToMap(data) ?: emptyMap())
        }
}

/** [Validation failure][ValidationFailure] wrapper for use from JavaScript. */
@JsExport
@JsName("ValidationFailure")
public class ValidationFailureJs internal constructor(issueKt: Any) : ValidationIssueJs() {
    override val issueKt: ValidationFailure =
        when (issueKt) {
            is ValidationFailure -> issueKt
            else -> error("Invalid argument.")
        }

    public val validation: String
        get() = issueKt.validation

    public val exception: String?
        get() = issueKt.exception

    public val stackTrace: String
        get() = issueKt.stackTrace
}

/** [Located validation issue][LocatedValidationIssue] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationIssue")
public sealed class LocatedValidationIssueJs {
    internal abstract val issueKt: LocatedValidationIssue

    public val path: AbsolutePathJs
        get() = issueKt.path.cachedToJs()

    public val code: String
        get() = issueKt.code

    public val dependencies: Array<AbsolutePathJs>
        get() = issueKt.dependencies.cachedToJs { it.cachedToJs() }

    public val dependsOnDescendants: Boolean
        get() = issueKt.dependsOnDescendants

    public val externalContextDependencies: Array<String>
        get() = issueKt.externalContextDependencies.cachedToJs()

    public val data: RecordTs<String, String?>
        get() = issueKt.data.cachedToJs()

    public val severity: ValidationIssueSeverityJs
        get() = issueKt.severity.toJs()

    override fun equals(other: Any?): Boolean =
        when {
            this === other -> true
            other !is LocatedValidationIssueJs -> false
            else -> issueKt == other.issueKt
        }

    override fun hashCode(): Int = issueKt.hashCode()

    public fun contains(issue: LocatedValidationIssueJs): Boolean = issueKt.contains(issue.issueKt)

    public override fun toString(): String = issueKt.toString()
}

internal fun LocatedValidationIssue.cachedToJs(): LocatedValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is LocatedValidationError -> LocatedValidationErrorJs(this)
            is LocatedValidationWarning -> LocatedValidationWarningJs(this)
            is LocatedValidationFailure -> LocatedValidationFailureJs(this)
        }
    }

/** Options available when building a [LocatedValidationErrorJs] or [LocatedValidationWarningJs]. */
public external interface LocatedValidationIssueOptionsJs {
    public val dependencies: Array<Any>?
    public val dependsOnDescendants: Boolean?
    public val externalContextDependencies: Array<String>?
    public val data: RecordTs<String, String?>?
}

/** [Located validation error][LocatedValidationError] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationError")
public class LocatedValidationErrorJs(
    issueKtOrPath: Any,
    code: String? = null,
    options: LocatedValidationIssueOptionsJs? = null,
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationError =
        issueKtOrPath as? LocatedValidationError
            ?: LocatedValidationError(
                issueKtOrPath.toPathKt(),
                code!!,
                options?.dependencies?.map { it.toPathKt() } ?: emptySet(),
                options?.dependsOnDescendants ?: false,
                options?.externalContextDependencies?.toSet() ?: emptySet(),
                jsObjectToMap(options?.data) ?: emptyMap(),
            )
}

/** [Located validation warning][LocatedValidationWarning] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationWarning")
public class LocatedValidationWarningJs(
    issueKtOrPath: Any,
    code: String? = null,
    options: LocatedValidationIssueOptionsJs? = null,
) : LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationWarning =
        issueKtOrPath as? LocatedValidationWarning
            ?: LocatedValidationWarning(
                issueKtOrPath.toPathKt(),
                code!!,
                options?.dependencies?.map { it.toPathKt() } ?: emptySet(),
                options?.dependsOnDescendants ?: false,
                options?.externalContextDependencies?.toSet() ?: emptySet(),
                jsObjectToMap(options?.data) ?: emptyMap(),
            )
}

/** [Located validation failure][LocatedValidationFailure] wrapper for use from JavaScript. */
@JsExport
@JsName("LocatedValidationFailure")
public class LocatedValidationFailureJs internal constructor(issueKt: Any) :
    LocatedValidationIssueJs() {
    override val issueKt: LocatedValidationFailure =
        when (issueKt) {
            is LocatedValidationFailure -> issueKt
            else -> error("Invalid argument.")
        }

    public val validation: String
        get() = issueKt.validation

    public val exception: String?
        get() = issueKt.exception

    public val stackTrace: String
        get() = issueKt.stackTrace
}

/**
 * Function which converts a [ValidationIssue] into a wrapped [ValidationIssueJs] object to be used
 * from JavaScript, while caching the conversion in the process.
 */
internal fun ValidationIssue.cachedToJs(): ValidationIssueJs =
    getOrSetFromCache(this) {
        when (this) {
            is ValidationError -> ValidationErrorJs(this)
            is ValidationWarning -> ValidationWarningJs(this)
            is ValidationFailure -> ValidationFailureJs(this)
        }
    }

/**
 * Function that converts a [LocatedValidationIssue] or a wrapped [LocatedValidationIssueJs] into a
 * [LocatedValidationIssue].
 */
internal fun Any.toLocatedValidationIssueKt(): LocatedValidationIssue =
    when (this) {
        is LocatedValidationIssue -> this
        is LocatedValidationIssueJs -> issueKt
        else -> error("Invalid located validation issue.")
    }
