import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.GlobalScope
import tech.ostack.kform.*

/** Options used when creating a schema from JavaScript. */
public external interface SchemaOptionsJs<T> {
    public val initialValue: T?
    public val validations: Array<ValidationJs<T>>?
}

/** [Schema] wrapper for use from JavaScript. */
@JsExport
@JsName("Schema")
public open class SchemaJs<T> internal constructor(internal val schemaKt: Schema<T>) {
    public val typeInfo: TypeInfoJs
        get() = schemaKt.typeInfo.cachedToJs()

    public val validations: Array<ValidationJs<T>>
        get() = schemaKt.validations.cachedToJs { it.cachedToJs() }

    public val initialValue: T
        get() = schemaKt.initialValue

    @OptIn(DelicateCoroutinesApi::class)
    public fun clone(value: T): CancellablePromise<T> =
        GlobalScope.cancellablePromise { schemaKt.clone(value) }
}

/** [Parent schema][ParentSchema] wrapper for use from JavaScript. */
@JsExport
@JsName("ParentSchema")
public open class ParentSchemaJs<T> internal constructor(schemaKt: ParentSchema<T>) :
    SchemaJs<T>(schemaKt)

/** [Collection schema][CollectionSchema] wrapper for use from JavaScript. */
@JsExport
@JsName("CollectionSchema")
public class CollectionSchemaJs<T> internal constructor(schemaKt: CollectionSchema<T, *>) :
    ParentSchemaJs<T>(schemaKt)

/**
 * Function which converts a [Schema] into a wrapped [SchemaJs] object to be used from JavaScript,
 * while caching the conversion in the process.
 */
public fun <T> Schema<T>.cachedToJs(): SchemaJs<T> =
    getOrSetFromCache(this) {
        when (this) {
            is CollectionSchema<*, *> -> CollectionSchemaJs(this as CollectionSchema<T, *>)
            is ParentSchema<*> -> ParentSchemaJs(this as ParentSchema<T>)
            else -> SchemaJs(this)
        }
    }

/** Function that returns a [Schema] from a wrapped [SchemaJs] or a [Schema]. */
internal fun Any.toSchemaKt(): Schema<*> =
    when (this) {
        is Schema<*> -> this
        is SchemaJs<*> -> schemaKt
        else -> error("Invalid schema.")
    }
